<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Form\Filter\Account;

use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\QueryBuilder;
use Ferienpass\AdminBundle\Form\Filter\AbstractFilterType;
use Ferienpass\CoreBundle\Entity\Edition;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Contracts\Translation\TranslatableInterface;

class EditionFilter extends AbstractFilterType
{
    public function __construct(private readonly Security $security)
    {
    }

    public function getParent(): string
    {
        return EntityType::class;
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'class' => Edition::class,
            'query_builder' => function (EntityRepository $er): QueryBuilder {
                $qb = $er->createQueryBuilder('e');

                if (!$this->security->isGranted('ROLE_ADMIN')) {
                    $qb->where('e.archived <> 1');
                }

                return $qb->orderBy('e.name');
            },
            'choice_value' => fn (?Edition $entity) => $entity?->getAlias(),
            'choice_label' => 'name',
            'placeholder' => '-',
            'multiple' => false,
        ]);
    }

    public function apply(QueryBuilder $qb, FormInterface $form): void
    {
        if ($form->isEmpty()) {
            return;
        }

        $k = $form->getName();
        $v = $form->getData();

        $expr = $qb->expr();

        $qb
            ->leftJoin('i.participants', 'p')
            ->leftJoin('a.offer', 'o_participants')
            ->leftJoin('i.hostAssociations', 'ha')
            ->leftJoin('ha.host', 'h')
            ->leftJoin('h.offers', 'o_hosts')
            ->andWhere($expr->orX(
                $expr->andX('o_hosts IS NOT NULL', 'o_hosts.edition = :q_'.$k),
                $expr->andX('o_participants IS NOT NULL', 'o_participants.edition = :q_'.$k)
            ))
            ->setParameter('q_'.$k, $v)
        ;
    }

    protected function getHumanReadableValue(FormInterface $form): string|TranslatableInterface|null
    {
        return $form->getData()?->getName();
    }
}
