<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Form;

use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\QueryBuilder;
use Ferienpass\CoreBundle\Entity\User;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\BirthdayType;
use Symfony\Component\Form\Extension\Core\Type\EmailType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

class EditParticipantType extends AbstractType
{
    public function __construct(#[Autowire(param: 'ferienpass.model.participant.class')] private readonly string $participantEntityClass)
    {
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefined('show_submit');

        $resolver->setDefaults([
            'data_class' => $this->participantEntityClass,
            'label_format' => 'participants.label.%name%',
            'translation_domain' => 'admin',
            'required' => false,
            'show_submit' => true,
        ]);
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder
            ->add('firstname', options: [
                'required' => true,
                'section' => 'default',
                'width' => '1/2',
            ])
            ->add('lastname', options: [
                'section' => 'default',
                'width' => '1/2',
            ])
            ->add('user', EntityType::class, options: [
                'class' => User::class,
                'query_builder' => fn (EntityRepository $er): QueryBuilder => $er->createQueryBuilder('user')
                    ->leftJoin('user.debtor', 'debtor')
                    ->leftJoin('user.blockedContact', 'blockedContact')
                    ->addSelect('debtor')
                    ->addSelect('blockedContact')
                    ->orderBy('user.lastname', 'ASC'),
                'autocomplete' => true,
                'choice_label' => fn (User $user): string => $user->getLastname().', '.$user->getFirstname(),
                'section' => 'default',
                'placeholder' => '-',
                'width' => '2/3',
                'help' => 'participants.help.user',
            ])
            ->add('dateOfBirth', BirthdayType::class, options: [
                'widget' => 'single_text',
                'section' => 'age',
                'width' => '1/3',
            ])
            ->add('ownEmail', EmailType::class, options: [
                'section' => 'contact',
                'width' => '1/2',
                'help' => 'participants.help.email',
            ])
            ->add('ownMobile', options: [
                'section' => 'contact',
                'width' => '1/2',
                'help' => 'participants.help.mobile',
            ])
            ->add('ownPhone', options: [
                'section' => 'contact',
                'width' => '1/2',
            ])
        ;

        if ($options['show_submit']) {
            $builder->add('submit', SubmitType::class, [
                'label' => 'Daten speichern',
            ]);
        }
    }
}
