<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Form;

use Ferienpass\AdminBundle\Form\Builder\DynamicFormBuilder;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Symfony\Bundle\SecurityBundle\Security;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;

class EditOfferType extends AbstractType
{
    public function __construct(private readonly Security $security, #[Autowire(param: 'ferienpass.model.offer.class')] private readonly string $offerEntityClass, private readonly OfferRepositoryInterface $offerRepository)
    {
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => $this->offerEntityClass,
            'is_variant' => false,
            'required' => false,
            'label_format' => 'offers.label.%name%',
            'translation_domain' => 'admin',
            'readonly' => false,
        ]);

        $resolver->addAllowedTypes('readonly', 'bool');
        $resolver->addAllowedTypes('is_variant', 'bool');
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder = DynamicFormBuilder::fromBuilder($builder) ?? new DynamicFormBuilder($builder, fn ($name) => $options['readonly'] || ($options['is_variant'] && \in_array($name, $this->offerRepository->variantFields(), true)));

        $builder
            ->add('name', options: [
                'section' => 'default',
                'width' => '2/3',
            ]);

        $builder->addIf('edition', condition: fn () => $this->security->isGranted('ROLE_ADMIN'), options: [
            'section' => 'default',
            'width' => '2/3',
        ]);

        $builder->addIf('hosts', condition: fn () => $this->security->isGranted('ROLE_ADMIN'), options: [
            'section' => 'default',
            'width' => '2/3',
        ])
            ->add('description', options: [
                'section' => 'default',
            ])
            ->add('applicationDeadline', options: [
                'section' => 'dates',
                'width' => '1/3',
            ])
            ->add('minAge', options: [
                'section' => 'details',
                'width' => '1/3',
            ])
            ->add('maxAge', options: [
                'section' => 'details',
                'width' => '1/3',
            ])
            ->add('meetingPoint', options: [
                'section' => 'details',
                'width' => '1/2',
            ])
            ->add('bring', options: [
                'section' => 'details',
                'width' => '1/2',
            ])
            ->add('fee', options: [
                'section' => 'details',
                'width' => '1/3',
            ])
            ->add('dates', options: [
                'section' => 'dates',
            ])
            ->add('wheelchairAccessible', options: [
                'section' => 'details',
                'width' => '1/3',
            ])
            ->add('requiresApplication', options: [
                'section' => 'applications',
                'width' => '1/2',
                'order' => 10,
            ])
            ->addIf('onlineApplication', dependencies: 'requiresApplication', options: [
                'section' => 'applications',
                'width' => '1/2',
                'order' => 12,
            ])
            ->addIf('applyText', dependencies: ['requiresApplication', 'onlineApplication'], condition: fn ($a, $b) => $a && !$b, options: [
                'section' => 'applications',
                'width' => '1/2',
                'order' => 14,
            ])
            ->add('requiresAgreementLetter', options: [
                'section' => 'agreementLetter',
                'width' => '1/2',
            ])
            ->add('consentForms', options: [
                'section' => 'agreementLetter',
            ])
            ->addIf('existingPdf', 'requiresAgreementLetter', null, OfferPdfType::class, ['section' => 'agreementLetter'])
            ->addIf('uploadAgreeLetter', 'requiresAgreementLetter', options: [
                'section' => 'agreementLetter',
            ])
            ->add('minParticipants', options: [
                'section' => 'applications',
                'width' => '1/3|clr',
                'order' => 20,
            ])
            ->add('maxParticipants', options: [
                'section' => 'applications',
                'width' => '1/3',
                'order' => 21,
            ])
            ->add('applicationExtra', options: [
                'section' => 'applicationExtra',
            ])
            ->add('contact', options: [
                'section' => 'contact',
            ])
            ->add('media', options: [
                'section' => 'media',
            ])
            ->add('uploadMedia', options: [
                'section' => 'media',
            ])
            ->add('attachments', options: [
                'section' => 'attachments',
            ])
            ->add('uploadAttachment', options: [
                'section' => 'attachments',
            ])
            ->add('submit', SubmitType::class)
        ;
    }
}
