<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Form\CompoundType;

use Contao\CoreBundle\Filesystem\VirtualFilesystemInterface;
use Contao\CoreBundle\Image\ImageFactoryInterface;
use Contao\Image\Exception\FileNotExistsException;
use Ferienpass\CoreBundle\Entity\OfferMedia;
use Imagine\Image\BoxInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\OptionsResolver;

class ImageUploadType extends AbstractType
{
    public function __construct(private readonly VirtualFilesystemInterface $mediaStorage, private readonly ImageFactoryInterface $imageFactory, #[Autowire(param: 'kernel.project_dir')] private readonly string $projectDir)
    {
    }

    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        $builder->addEventListener(FormEvents::PRE_SET_DATA, function (FormEvent $event): void {
            $form = $event->getForm();
            $data = $event->getData();

            if (!$data instanceof OfferMedia) {
                return;
            }

            try {
                $dimensions = $this->imageFactory->create(\sprintf('%s/%s/%s', $this->projectDir, $this->mediaStorage->getPrefix(), $this->mediaStorage->get($data->getMedia()->getUuid())->getPath()))->getDimensions();
            } catch (FileNotExistsException) {
                return;
            }

            $transformImportantPart = function (array $importantPart, BoxInterface $size) {
                if ([] === array_filter($importantPart)) {
                    return null;
                }

                return [
                    'width' => $importantPart['width'] * $size->getWidth(),
                    'height' => $importantPart['height'] * $size->getHeight(),
                    'x' => $importantPart['x'] * $size->getWidth(),
                    'y' => $importantPart['y'] * $size->getHeight(),
                ];
            };

            $form
                ->add('media', CropperType::class, [
                    'label' => false,
                    'public_url' => (string) $this->mediaStorage->generatePublicUri($data->getMedia()->getUuid()),
                    'dimensions' => $dimensions->getSize(),
                    'cropper_options' => [
                        'zoomOnTouch' => false,
                        'zoomOnWheel' => false,
                        'guides' => false,
                        'autoCropArea' => 1.0,
                        'data' => $transformImportantPart($data->getMedia()->getImportantPart(), $dimensions->getSize()),
                    ],
                ])
            ;

            $form
                ->add('alt', TextType::class, [
                    'help' => 'Hier geben Sie an, was auf dem Bild zu sehen ist (Barrierefreiheit)',
                    'property_path' => 'media.alt',
                ])
                ->add('creator', TextType::class, [
                    'required' => true,
                    'help' => 'Hier geben Sie den Namen des/der Fotograf:in oder den Namen des/der Ersteller:in des Kunstwerkes (Urheberrecht)',
                    'property_path' => 'media.creator',
                ])
            ;
        });
    }

    public function configureOptions(OptionsResolver $resolver): void
    {
        $resolver->setDefaults([
            'data_class' => OfferMedia::class,
        ]);
    }
}
