<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Form\CompoundType;

use Ferienpass\CoreBundle\Entity\OfferMedia;
use Imagine\Image\BoxInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Form\FormView;
use Symfony\Component\OptionsResolver\OptionsResolver;

class CropperType extends AbstractType
{
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            ->add('importantPart', HiddenType::class, [
                'error_bubbling' => true,
                'attr' => [
                    'data-controller' => trim(($options['attr']['data-controller'] ?? '').' symfony--ux-cropperjs--cropper'),
                    'data-symfony--ux-cropperjs--cropper-public-url-value' => $options['public_url'],
                    'data-symfony--ux-cropperjs--cropper-options-value' => json_encode($options['cropper_options']),
                ],
                'getter' => fn () => null,
                'setter' => function (OfferMedia &$media, ?string $value) use ($options): void {
                    $data = json_decode($value ?? '', true);
                    if (!$data) {
                        return;
                    }

                    $imgWidth = $options['dimensions']->getWidth();
                    $imgHeight = $options['dimensions']->getHeight();

                    // Normalize to [0,1]
                    $x = max(0.0, min(1.0, ($data['x'] ?? 0) / $imgWidth));
                    $y = max(0.0, min(1.0, ($data['y'] ?? 0) / $imgHeight));
                    $width = max(0.0, min(1.0, ($data['width'] ?? 0) / $imgWidth));
                    $height = max(0.0, min(1.0, ($data['height'] ?? 0) / $imgHeight));

                    // Ensure the important part stays within bounds
                    $width = min($width, max(0.0, 1.0 - $x));
                    $height = min($height, max(0.0, 1.0 - $y));

                    // Round (towards zero to keep values conservative)
                    $x = round($x, 3, \RoundingMode::TowardsZero);
                    $y = round($y, 3, \RoundingMode::TowardsZero);
                    $width = round($width, 3, \RoundingMode::TowardsZero);
                    $height = round($height, 3, \RoundingMode::TowardsZero);

                    // Re-apply bounds after rounding to avoid x+width or y+height exceeding 1 due to precision
                    $width = min($width, max(0.0, 1.0 - $x));
                    $height = min($height, max(0.0, 1.0 - $y));

                    $media->getMedia()->setImportantPart([
                        'width' => $width,
                        'height' => $height,
                        'x' => $x,
                        'y' => $y,
                    ]);
                },
            ])
        ;
    }

    public function buildView(FormView $view, FormInterface $form, array $options)
    {
        // Remove data-controller attribute as it's already on the child input
        if (isset($view->vars['attr']['data-controller'])) {
            unset($view->vars['attr']['data-controller']);
        }
    }

    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setRequired('public_url');
        $resolver->setRequired('dimensions');
        $resolver->setAllowedTypes('public_url', 'string');
        $resolver->setAllowedTypes('dimensions', BoxInterface::class);
        $resolver->setDefault('cropper_options', []);

        $resolver->setDefaults([
            'label' => false,
            'inherit_data' => true,
        ]);
    }

    public function getBlockPrefix()
    {
        return 'cropper';
    }
}
