<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Statistics;

use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Contracts\Translation\TranslatorInterface;

class ChartUtilizationController extends AbstractController
{
    public function __construct(private readonly AttendanceRepository $attendances, private readonly TranslatorInterface $translator)
    {
    }

    public function __invoke(Edition $edition): Response
    {
        return $this->render('@FerienpassAdmin/fragment/statistics/chart_utilization.html.twig', [
            'data' => $this->getData($edition->getId()),
        ]);
    }

    private function getData(int $passEdition): array
    {
        $return = [];
        $result = $this->attendances->createQueryBuilder('a')
            ->select('a.status')
            ->addSelect('o.id as offer_id')
            ->addSelect('o.name as offer_title')
            ->addSelect('MIN(d.begin) as date_start')
            ->addSelect('o.maxParticipants as offer_max')
            ->addSelect('COUNT(a.id) as count_applications')
            ->addSelect('COUNT(a.id) / o.maxParticipants as utilization')
            ->innerJoin('a.offer', 'o')
            ->leftJoin('o.dates', 'd')
            ->where('o.edition = :edition')
            ->setParameter('edition', $passEdition)
            ->groupBy('o.id', 'a.status')
            ->orderBy('utilization', 'DESC')
            ->getQuery()
            ->getScalarResult()
        ;

        $labels = [];
        $overall = [];

        $confirmed = [];
        $waitlisted = [];
        $error = [];
        $waiting = [];

        $offerIds = array_values(array_unique(array_column($result, 'offer_id')));

        foreach ([Attendance::STATUS_CONFIRMED, Attendance::STATUS_WAITLISTED, Attendance::STATUS_REJECTED, Attendance::STATUS_UNFULFILLED, Attendance::STATUS_WAITING] as $status) {
            $utilizationOfStatus = array_filter($result, fn ($c) => $status === $c['status']);

            foreach ($offerIds as $i => $offerId) {
                $utilizationOfStatusAndOffer = array_values(array_filter($utilizationOfStatus, fn ($c) => (int) $c['offer_id'] === (int) $offerId));
                if ([] !== $utilizationOfStatusAndOffer && null !== ($a = ($utilizationOfStatusAndOffer[0] ?? null)) && !$a['date_start'] instanceof \DateTimeInterface) {
                    ${$status}[$i] = (float) $a['utilization'];
                    $overall[$i] = ($overall[$i] ?? 0) + (float) $a['utilization'];

                    $labels[$i] = \sprintf('%s: %s (max. %d)', $a['offer_title'], $a['date_start'] ? new \DateTime($a['date_start'])->format('d.m.Y') : '', $a['offer_max']);
                } else {
                    ${$status}[$i] = 0;
                    $overall[$i] = ($overall[$i] ?? 0) + 0;

                    if (!isset($labels[$i])) {
                        $labels[$i] = '';
                    }

                    if (!isset($overall[$i])) {
                        $overall[$i] = 0;
                    }
                }
            }
        }

        $return['labels'] = array_values($labels);
        foreach ([Attendance::STATUS_CONFIRMED, Attendance::STATUS_WAITLISTED, Attendance::STATUS_REJECTED,  Attendance::STATUS_UNFULFILLED, Attendance::STATUS_WAITING] as $status) {
            if (!${$status}) {
                continue;
            }

            $return['datasets'][] = [
                'name' => $this->translator->trans('status.'.$status),
                'values' => array_values(${$status}),
            ];
        }

        return $return;
    }
}
