<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Statistics;

use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;

class ChartAgeController extends AbstractController
{
    public function __construct(private readonly AttendanceRepository $attendances)
    {
    }

    public function __invoke(Edition $edition): Response
    {
        return $this->render('@FerienpassAdmin/fragment/statistics/chart_age.html.twig', [
            'data' => $this->getData($edition->getId()),
        ]);
    }

    private function getData(int $passEdition): array
    {
        $ageAndCount = $this->attendances->createQueryBuilder('a')
            ->select(
                <<<'SQL'
                    (
                        CASE
                            WHEN (p.dateOfBirth IS NULL AND a.age IS NOT NULL) THEN a.age
                            WHEN (p.dateOfBirth IS NOT NULL AND d.begin IS NOT NULL) THEN TIMESTAMPDIFF(YEAR, p.dateOfBirth, MIN(d.begin))
                            ELSE 'N/A' END
                        ) as age
                    SQL
            )
            ->addSelect('COUNT(a.id) as count')
            ->innerJoin('a.offer', 'o')
            ->leftJoin('a.participant', 'p')
            ->leftJoin('o.dates', 'd')
            ->where('o.edition = :edition')
            ->andWhere('a.status <> :status')
            ->setParameter('edition', $passEdition)
            ->setParameter('status', Attendance::STATUS_WITHDRAWN)
            ->groupBy('o.id')
            ->addGroupBy('age')
            ->getQuery()
            ->getScalarResult()
        ;

        // Transform array to key=>value structure
        $ageAndCount = array_combine(array_column($ageAndCount, 'age'), array_column($ageAndCount, 'count'));
        ksort($ageAndCount, \SORT_NATURAL);

        $return = [];
        foreach ($ageAndCount as $age => $count) {
            $return[] = [
                'title' => is_numeric($age) ? \sprintf('%d Jahre', $age) : $age,
                'count' => (int) $count,
            ];
        }

        return $return;
    }
}
