<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\AdminBundle\Form\EditAccessCodesType;
use Ferienpass\CoreBundle\Entity\AccessCodeStrategy;
use Ferienpass\CoreBundle\Entity\MessengerLog;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Facade\EraseDataFacade;
use Ferienpass\CoreBundle\Repository\AccessCodeStrategyRepository;
use Ferienpass\CoreBundle\Repository\HostConsentRepository;
use Ferienpass\CoreBundle\Session\Flash;
use Symfony\Bridge\Doctrine\Attribute\MapEntity;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\CurrentUser;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\UX\Turbo\TurboBundle;

final class ToolsController extends AbstractController
{
    public function __construct()
    {
    }

    #[Route('/tools', name: 'admin_tools')]
    #[IsGranted('ROLE_ADMIN')]
    public function index(Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/tools/index.html.twig', [
            'breadcrumb' => $breadcrumb->generate('tools.title'),
        ]);
    }

    #[IsGranted('ROLE_ADMIN')]
    #[Route('/export', name: 'admin_export_index')]
    public function export(Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/tools/export.html.twig', [
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'export.title'),
        ]);
    }

    #[Route('/rundmail', name: 'admin_tools_mailing')]
    public function mailing(#[CurrentUser] User $user, Breadcrumb $breadcrumb, HostConsentRepository $consents, #[Autowire(param: 'ferienpass_admin.privacy_consent_text')] string $consentText): Response
    {
        if ('' !== $consentText && !$consents->findValid($user) instanceof \Ferienpass\CoreBundle\Entity\HostConsent) {
            return $this->render('@FerienpassAdmin/page/missing_privacy_statement.html.twig', [
                'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'mailing.title'),
            ]);
        }

        return $this->render('@FerienpassAdmin/page/tools/mailing.html.twig', [
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'mailing.title'),
        ]);
    }

    #[Route('/versand/{uuid}', name: 'admin_tools_dispatch')]
    public function dispatch(#[MapEntity(mapping: ['uuid' => 'uuid'])] MessengerLog $log, Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/tools/dispatch.html.twig', [
            'log' => $log,
            'breadcrumb' => $breadcrumb->generate('dispatch.title'),
        ]);
    }

    #[Route('/betroffenenrechte', name: 'admin_tools_subjectrights')]
    #[IsGranted('ROLE_SUPER_ADMIN')]
    public function subjectRights(Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/tools/noop.html.twig', [
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'subjectrichts.title'),
        ]);
    }

    #[Route('/system', name: 'admin_tools_settings')]
    #[IsGranted('ROLE_SUPER_ADMIN')]
    public function settings(): Response
    {
        return $this->redirectToRoute('admin_tools_settings_categories');
    }

    #[Route('/system/angebotskategorien', name: 'admin_tools_settings_categories')]
    #[IsGranted('ROLE_SUPER_ADMIN')]
    public function settingsCategories(Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/tools/settings_categories.html.twig', [
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'settings.title', 'offerCategories.title'),
        ]);
    }

    #[IsGranted('ROLE_SUPER_ADMIN')]
    #[Route('/postausgang', name: 'admin_outbox')]
    public function outbox(Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/tools/outbox.html.twig', [
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'outbox.title'),
        ]);
    }

    #[IsGranted('ROLE_SUPER_ADMIN')]
    #[Route('/daten-löschen', name: 'admin_erase_data')]
    public function eraseData(EraseDataFacade $eraseDataFacade, Breadcrumb $breadcrumb, Request $request): Response
    {
        $participants = $eraseDataFacade->expiredParticipants();

        $form = $this->createFormBuilder()->getForm();
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $eraseDataFacade->eraseData();

            return $this->redirectToRoute('admin_erase_data');
        }

        return $this->render('@FerienpassAdmin/page/tools/erase_data.html.twig', [
            'form' => $form,
            'participants' => $participants,
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'eraseData.title'),
        ]);
    }

    #[Route('/system/zugangscodes', name: 'admin_accessCodes_index')]
    public function accessCodes(Breadcrumb $breadcrumb, AccessCodeStrategyRepository $repository): Response
    {
        $qb = $repository->createQueryBuilder('i');

        $qb->addOrderBy('i.name', 'ASC');

        return $this->render('@FerienpassAdmin/page/tools/access_codes.html.twig', [
            'qb' => $qb,
            'createUrl' => $this->generateUrl('admin_accessCodes_create'),
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'settings.title', 'accessCodes.title'),
        ]);
    }

    #[Route('/einstellungen/zugangscodes/neu', name: 'admin_accessCodes_create')]
    #[Route('/einstellungen/zugangscodes/{id}', name: 'admin_accessCodes_edit')]
    public function edit(?AccessCodeStrategy $accessCodeStrategy, Request $request, EntityManagerInterface $em, Breadcrumb $breadcrumb, Flash $flash): Response
    {
        $accessCodeStrategy ??= new AccessCodeStrategy();
        $em->persist($accessCodeStrategy);

        $form = $this->createForm(EditAccessCodesType::class, $accessCodeStrategy);

        $breadcrumbTitle = $accessCodeStrategy ? $accessCodeStrategy->getName().' (bearbeiten)' : 'accessCodes.new';

        /** @noinspection FormViewTemplate `createView()` messes ups error handling/redirect */
        return $this->render('@FerienpassAdmin/page/tools/access_codes_edit.html.twig', [
            'item' => $accessCodeStrategy,
            'form' => $form,
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'settings.title', 'accessCodes.title', $breadcrumbTitle),
        ]);
    }

    #[Route('/einstellungen/zugangscodes/{id}/löschen', name: 'admin_accessCodes_delete', requirements: ['id' => '\d+'])]
    public function delete(AccessCodeStrategy $item, Request $request, EntityManagerInterface $entityManager): Response
    {
        $this->denyAccessUnlessGranted('delete', $item);

        $form = $this->createForm(FormType::class, options: [
            'action' => $this->generateUrl('admin_participants_delete', ['id' => $item->getId()]),
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $deletedId = $item->getId();

            $entityManager->remove($item);
            $entityManager->flush();

            if (TurboBundle::STREAM_FORMAT === $request->getPreferredFormat()) {
                // If the request comes from Turbo, set the content type as text/vnd.turbo-stream.html and only send the HTML to update
                $request->setRequestFormat(TurboBundle::STREAM_FORMAT);

                return $this->renderBlock('@FerienpassAdmin/page/tools/access_codes.html.twig', 'deleted_stream', ['id' => $deletedId]);
            }

            return $this->redirectToRoute('admin_accessCodes_index');
        }

        return $this->render('@FerienpassAdmin/page/tools/access_codes_delete.html.twig', [
            'item' => $item,
            'form' => $form,
        ]);
    }

    #[Route('/tools/schnellerfassung', name: 'admin_tools_registration')]
    #[IsGranted('ROLE_ADMIN')]
    public function quickRegistration(Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/tools/quick_registration.html.twig', [
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'Schnell-Erfassung'),
        ]);
    }

    #[Route('/tools/problem-melden', name: 'admin_tools_issue')]
    #[IsGranted('ROLE_ADMIN')]
    public function createIssue(Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/tools/create_issue.html.twig', [
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'Ticket erstellen'),
        ]);
    }
}
