<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CoreBundle\Entity\ApiToken;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Generator\ApiTokenGenerator;
use Ferienpass\CoreBundle\Session\Flash;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\CurrentUser;
use Symfony\Component\Translation\TranslatableMessage;

#[Route('/sicherheit')]
final class SecurityController extends AbstractController
{
    private const string TMP_SESSION_KEY = 'admin_security_api_token:value';

    #[Route('', name: 'admin_security')]
    public function __invoke(EntityManagerInterface $em, Request $request, Flash $flash): Response
    {
        /** @noinspection FormViewTemplate `createView()` messes ups error handling/redirect */
        return $this->render('@FerienpassAdmin/page/user/security.html.twig');
    }

    #[Route('/api-token', name: 'admin_security_api_token')]
    public function apiToken(#[CurrentUser] User $user, Request $request, EntityManagerInterface $entityManager, ApiTokenGenerator $apiTokenGenerator, UserPasswordHasherInterface $passwordHasher): Response
    {
        $session = $request->getSession();
        $fb = $this->createFormBuilder(options: [
            'action' => $this->generateUrl('admin_security_api_token'),
        ]);

        $fb->add('api:offer', ChoiceType::class, [
            'choices' => [
                'api:offer:read',
                // 'api:offer:write',
            ],
            'choice_label' => fn ($choice): TranslatableMessage => new TranslatableMessage($choice, domain: 'admin'),
            'expanded' => true,
            'required' => false,
            'multiple' => false,
            'label' => 'Angebote',
        ]);

        $form = $fb->getForm();

        if ($session->has(self::TMP_SESSION_KEY)) {
            $apiToken = $session->get(self::TMP_SESSION_KEY);
            $session->remove(self::TMP_SESSION_KEY);

            $token = $entityManager->getRepository(ApiToken::class)->findOneBy(['locator' => explode('.', (string) $apiToken)[0]]);

            return $this->render('@FerienpassAdmin/page/security/api-token.html.twig', [
                'token' => $token,
                'apiToken' => $apiToken,
            ]);
        }

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $apiToken = $apiTokenGenerator->generate();
            $session->set(self::TMP_SESSION_KEY, $apiToken);

            $scopes = array_values(array_filter($form->getData(), fn ($v) => str_starts_with((string) $v, 'api:')));
            $token = ApiToken::fromToken($apiToken, $scopes, $user, $passwordHasher);

            $entityManager->persist($token);
            $entityManager->flush();

            return $this->redirectToRoute('admin_security_api_token');
        }

        return $this->render('@FerienpassAdmin/page/security/api-token.html.twig', [
            'form' => $form,
        ]);
    }
}
