<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\AdminBundle\Export\ExportQueryBuilderInterface;
use Ferienpass\AdminBundle\Form\Filter\PaymentsFilter;
use Ferienpass\AdminBundle\Form\PaymentReverseType;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Debtor;
use Ferienpass\CoreBundle\Entity\Payment;
use Ferienpass\CoreBundle\Entity\PaymentItem;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Export\Payments\ReceiptPdfExport;
use Ferienpass\CoreBundle\Facade\PaymentsFacade;
use Ferienpass\CoreBundle\Message\ReorderParticipantList;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Ferienpass\CoreBundle\Repository\BookEntryRepository;
use Ferienpass\CoreBundle\Repository\DebtorRepository;
use Ferienpass\CoreBundle\Repository\PaymentRepository;
use Ferienpass\CoreBundle\Session\Flash;
use Psr\EventDispatcher\EventDispatcherInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\TaggedLocator;
use Symfony\Component\DependencyInjection\ServiceLocator;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\CurrentUser;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Workflow\WorkflowInterface;

#[IsGranted('ROLE_ADMIN')]
#[Route('/zahlungen')]
final class PaymentsController extends AbstractController
{
    public function __construct(private readonly ReceiptPdfExport $receiptExport, private readonly Flash $flash, private readonly EntityManagerInterface $entityManager, private readonly MessageBusInterface $commandBus, #[TaggedLocator(ExportQueryBuilderInterface::class, defaultIndexMethod: 'getFormat')] private readonly ServiceLocator $exporters, private readonly WorkflowInterface $paymentStateMachine)
    {
    }

    #[Route('', name: 'admin_payments_index')]
    public function index(PaymentRepository $repository, Breadcrumb $breadcrumb): Response
    {
        $qb = $repository->createQueryBuilder('i')
            ->where('i.status <> :status_abandoned')
            ->setParameter('status_abandoned', Payment::STATUS_ABANDONED)
            ->orderBy('i.createdAt', 'DESC')
        ;

        return $this->render('@FerienpassAdmin/page/payments/index.html.twig', [
            'qb' => $qb,
            'filterType' => PaymentsFilter::class,
            'exports' => array_keys($this->exporters->getProvidedServices()),
            'searchable' => ['billingAddress', 'billingEmail', 'receiptNumber'],
            'breadcrumb' => $breadcrumb->generate('payments.title'),
        ]);
    }

    #[Route('/dashboard', name: 'admin_payments_dashboard')]
    public function dashboard(BookEntryRepository $bookEntryRepository, DebtorRepository $debtorsRepository, Breadcrumb $breadcrumb, EventDispatcherInterface $dispatcher): Response
    {
        $sumByDay = $bookEntryRepository->createQueryBuilder('entry')
            ->select('SUM(entry.value) as sum')
            ->addSelect("DATE_FORMAT(entry.bookingDate, '%Y-%m-%d') AS date")
            ->where('entry.type = :type')
            ->setParameter('type', 'payment')
            ->groupBy('date')
            ->orderBy('date', 'ASC')
            ->getQuery()
            ->getResult()
        ;

        $chart = [];
        foreach (new \DatePeriod(new \DateTime('-1 year'), new \DateInterval('P1D'), new \DateTime('tomorrow')) as $date) {
            $chart['labels'][] = $date->format('d.m.Y');
            $entry = array_find($sumByDay, fn (array $v) => $v['date'] === $date->format('Y-m-d'));
            $chart['values'][] = $entry ? (int) ($entry['sum']) / 100 : 0;
        }

        $qb = $debtorsRepository->createQueryBuilder('debtor')
            ->leftJoin('debtor.user', 'user')
        ;

        // Solve N+1
        $qb
            ->addSelect('user')
            ->leftJoin('user.participants', 'participant')
            ->addSelect('participant')
            ->leftJoin('participant.attendances', 'attendance')
            ->addSelect('attendance')
            ->leftJoin('debtor.bookEntries', 'entry')
            ->addSelect('entry')
        ;

        return $this->render('@FerienpassAdmin/page/payments/dashboard/debtors.html.twig', [
            'qb' => $qb,
            'searchable' => ['user.firstname', 'user.lastname'],
            'chart' => $chart,
            'dispatcher' => $dispatcher,
            'breadcrumb' => $breadcrumb->generate('payments.title'),
        ]);
    }

    #[Route('/dashboard/ausstehend', name: 'admin_payments_unpaid')]
    public function unpaid(AttendanceRepository $attendances, Breadcrumb $breadcrumb, EventDispatcherInterface $dispatcher): Response
    {
        $qb = $attendances->createQueryBuilder('attendance')
            ->innerJoin('attendance.offer', 'offer')
            ->innerJoin('attendance.participant', 'participant')
            ->leftJoin('attendance.activity', 'activity')
            ->leftJoin('activity.createdBy', 'activityUser')
            ->addSelect('participant')
            ->addSelect('offer')
            ->where('offer.fee > 0')
        ;

        PaymentsFacade::addWhereLogic($qb);

        return $this->render('@FerienpassAdmin/page/payments/dashboard/unpaid.html.twig', [
            'qb' => $qb,
            'searchable' => ['offer.name', 'participant.firstname', 'participant.lastname'],
            'dispatcher' => $dispatcher,
            'breadcrumb' => $breadcrumb->generate('payments.title'),
        ]);
    }

    #[Route('/dashboard/überzahlt', name: 'admin_payments_overpaid')]
    public function overpaid(AttendanceRepository $attendances, Breadcrumb $breadcrumb, EventDispatcherInterface $dispatcher): Response
    {
        $qb = $attendances->createQueryBuilder('a')
            ->innerJoin('a.paymentItems', 'pi')
            ->innerJoin('pi.payment', 'p')
            ->leftJoin('a.offer', 'offer')
            ->leftJoin('a.participant', 'participant')
            ->groupBy('a.id')
            ->where('a.payable = 1')
            ->andWhere('p.status = :paid')
            ->setParameter('paid', Payment::STATUS_PAID)
            ->having('SUM(CASE WHEN pi.amount > 0 THEN 1 ELSE 0 END) - SUM(CASE WHEN pi.amount < 0 THEN 1 ELSE 0 END) > 2')
        ;

        return $this->render('@FerienpassAdmin/page/payments/dashboard/overpaid.html.twig', [
            'qb' => $qb,
            'searchable' => ['offer.name', 'participant.firstname', 'participant.lastname'],
            'dispatcher' => $dispatcher,
            'breadcrumb' => $breadcrumb->generate('payments.title'),
        ]);
    }

    #[Route('/debitoren/{id}', name: 'admin_payments_debtor')]
    public function debtor(Debtor $debtor, Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/payments/debtor.html.twig', [
            'debtor' => $debtor,
            'breadcrumb' => $breadcrumb->generate('payments.title'),
        ]);
    }

    #[Route('/debitoren/{id}/freie-buchung', name: 'admin_payments_create_booking')]
    public function createBooking(Debtor $debtor, Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/payments/createBooking.html.twig', [
            'debtor' => $debtor,
            'breadcrumb' => $breadcrumb->generate('payments.title'),
        ]);
    }

    #[Route('/export.{format}', name: 'admin_payments_export', requirements: ['format' => '\w+'])]
    public function export(PaymentRepository $repository, string $format)
    {
        $qb = $repository->createQueryBuilder('i');
        $qb->orderBy('i.createdAt', 'DESC');

        if (!$this->exporters->has($format)) {
            throw $this->createNotFoundException();
        }

        $exporter = $this->exporters->get($format);

        return $this->file($exporter->generate($qb), "zahlungen.$format");
    }

    #[Route('/{id}', name: 'admin_payments_receipt', requirements: ['id' => '\d+'])]
    public function show(Payment $payment, Breadcrumb $breadcrumb): Response
    {
        return $this->render('@FerienpassAdmin/page/payments/receipt.html.twig', [
            'payment' => $payment,
            'breadcrumb' => $breadcrumb->generate(['payments.title', ['route' => 'admin_payments_index']], 'Beleg #'.$payment->getReceiptNumber()),
        ]);
    }

    #[Route('/{id}/storno', name: 'admin_payments_reverse')]
    public function reverse(#[CurrentUser] User $user, Payment $payment, Request $request, Breadcrumb $breadcrumb): Response
    {
        $form = $this->createForm(PaymentReverseType::class, options: [
            'action' => $this->generateUrl('admin_payments_reverse', ['id' => $payment->getId()]),
            'payment' => $payment,
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            /** @var ArrayCollection<int, PaymentItem> $items */
            $items = $form->get('items')->getData();
            $items = $items->filter(fn (PaymentItem $i) => false !== $i->getAttendance()?->isPaid());

            if ($items->isEmpty()) {
                $this->flash->addError(text: 'Es wurde nichts storniert. Entweder wurde keine Auswahl getroffen, oder die Buchungen waren schon storniert.');

                return $this->redirectToRoute('admin_payments_index');
            }

            $debtor = $payment->getDebtor();
            // Legacy
            if (!$debtor instanceof Debtor) {
                $debtor = array_unique(array_filter(array_map(fn (PaymentItem $a) => $a->getAttendance()->getParticipant()->getUser()?->getDebtor(), $items->toArray())))[0] ?? null;
            }

            $reversalPayment = new Payment($debtor, user: $user);
            $reversalPayment->setBillingAddress($payment->getBillingAddress());
            $reversalPayment->setBillingEmail($payment->getBillingEmail());
            foreach ($items as $item) {
                $reversalPayment->addItem(new PaymentItem(
                    $reversalPayment,
                    (-1) * $item->getAmount(),
                    $item->getAttendance()
                ));
            }

            $this->paymentStateMachine->apply($reversalPayment, Payment::TRANSITION_PAY, [
                'reverse' => $form->getData()['unpay'] ?? null,
                'payout' => $form->getData()['payout'] ?? null,
            ]);

            if ($form->getData()['withdraw'] ?? false) {
                foreach ($reversalPayment->getItems() as $item) {
                    if ($item->getAttendance()->isWithdrawn()) {
                        continue;
                    }

                    $item->getAttendance()->setStatus(Attendance::STATUS_WITHDRAWN);

                    $this->commandBus->dispatch(new ReorderParticipantList($item->getAttendance()->getOffer()->getId()));
                }
            }

            $this->entityManager->persist($reversalPayment);
            $this->entityManager->flush();

            $this->flash->addConfirmation(text: 'Der Stornobeleg wurde erstellt.');

            return $this->redirectToRoute('admin_payments_receipt', ['id' => $reversalPayment->getId()]);
        }

        return $this->render('@FerienpassAdmin/page/payments/reverse.html.twig', [
            'form' => $form,
            'payment' => $payment,
            'breadcrumb' => $breadcrumb->generate(['payments.title', ['route' => 'admin_payments_index']], 'Beleg #'.$payment->getReceiptNumber(), 'Storno'),
        ]);
    }

    #[Route('/{id}.pdf', name: 'admin_payments_receipt_pdf', requirements: ['id' => '\d+'])]
    public function pdf(Payment $payment): Response
    {
        $path = $this->receiptExport->generate($payment);

        return $this->file($path, \sprintf('beleg-%s.pdf', $payment->getId()));
    }
}
