<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ManagerRegistry;
use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\AdminBundle\Dto\BillingAddressDto;
use Ferienpass\AdminBundle\Export\ExportQueryBuilderInterface;
use Ferienpass\AdminBundle\Export\XlsxExportQueryBuilder;
use Ferienpass\AdminBundle\Form\EditParticipantType;
use Ferienpass\AdminBundle\Form\Filter\AttendancesFilter;
use Ferienpass\AdminBundle\Form\Filter\ParticipantFilter;
use Ferienpass\AdminBundle\Form\FormScreenFactory;
use Ferienpass\AdminBundle\Form\HandleFormTrait;
use Ferienpass\AdminBundle\Form\SettleAttendancesType;
use Ferienpass\AdminBundle\LiveComponent\MultiSelect;
use Ferienpass\AdminBundle\LiveComponent\MultiSelectHandlerInterface;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Debtor;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;
use Ferienpass\CoreBundle\Entity\Payment;
use Ferienpass\CoreBundle\Entity\PaymentItem;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Ferienpass\CoreBundle\Repository\ParticipantRepositoryInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\TaggedLocator;
use Symfony\Component\DependencyInjection\ServiceLocator;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Attribute\MapQueryParameter;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Requirement\Requirement;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\Component\Workflow\WorkflowInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Symfony\UX\Turbo\TurboBundle;

#[IsGranted('ROLE_PARTICIPANTS_ADMIN')]
#[Route('/teilnehmende')]
final class ParticipantsController extends AbstractController implements MultiSelectHandlerInterface
{
    use HandleFormTrait;

    public function __construct(private readonly ManagerRegistry $doctrine, #[TaggedLocator(ExportQueryBuilderInterface::class, defaultIndexMethod: 'getFormat')] private readonly ServiceLocator $exporters)
    {
    }

    #[Route('', name: 'admin_participants_index')]
    public function index(ParticipantRepositoryInterface $repository, Breadcrumb $breadcrumb, ?string $_suffix, XlsxExportQueryBuilder $xlsxExport): Response
    {
        $qb = $repository->createQueryBuilder('i');

        // Solve N+1
        $qb
            ->leftJoin('i.attendances', 'a')
            ->addSelect('a')
            ->leftJoin('a.offer', 'o')
            ->addSelect('o')
            ->leftJoin('i.activity', 'l')
            ->addSelect('l')
        ;

        // $filter = $this->filterFactory->create($qb)->applyFilter($request->query->all());

        return $this->render('@FerienpassAdmin/page/participants/index.html.twig', [
            'qb' => $qb,
            'filterType' => ParticipantFilter::class,
            'exports' => array_keys($this->exporters->getProvidedServices()),
            'searchable' => ['firstname', 'lastname', 'email', 'mobile', 'phone'],
            'createUrl' => $this->generateUrl('admin_participants_create'),
            'breadcrumb' => $breadcrumb->generate('participants.title'),
        ]);
    }

    #[Route('/export.{format}', name: 'admin_participants_export', requirements: ['format' => '\w+'])]
    public function export(ParticipantRepositoryInterface $repository, string $format)
    {
        $qb = $repository->createQueryBuilder('i');

        if (!$this->exporters->has($format)) {
            throw $this->createNotFoundException();
        }

        $exporter = $this->exporters->get($format);

        return $this->file($exporter->generate($qb), "teilnehmende.$format");
    }

    #[Route('/neu/{section?default}', name: 'admin_participants_create')]
    #[Route('/{uuid}/{section?default}', name: 'admin_participants_edit', requirements: ['uuid' => Requirement::UUID], priority: -1)]
    public function edit(?string $uuid, string $section, Request $request, \Symfony\Component\EventDispatcher\EventDispatcherInterface $dispatcher, EntityManagerInterface $entityManager, Breadcrumb $breadcrumb, ParticipantRepositoryInterface $repository, EntityManagerInterface $em, FormScreenFactory $forms): Response
    {
        if (null !== $uuid && !($entity = $repository->findByUuid($uuid)) instanceof ParticipantInterface) {
            throw $this->createNotFoundException();
        }

        $entity ??= $repository->createNew();

        $form = $this->createForm(EditParticipantType::class, $entity, $formOptions = [
            'display_section' => $section,
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $this->handleSubmit($form, $dispatcher, $request);

            $entityManager->flush();

            return $this->redirectToRoute('admin_participants_edit', ['uuid' => $entity->getUuid(), 'section' => $section]);
        }

        $breadcrumbTitle = $entity ? $entity->getName().' (bearbeiten)' : 'participants.new';

        return $this->render('@FerienpassAdmin/page/participants/edit.html.twig', [
            'item' => $entity,
            'form' => $form,
            'formOptions' => $formOptions,
            'formScreen' => $forms->screen(EditParticipantType::class, $entity),
            'section' => $section,
            'breadcrumb' => $breadcrumb->generate(['participants.title', ['route' => 'admin_participants_index']], $breadcrumbTitle),
        ]);
    }

    #[Route('/{uuid}/info', name: 'admin_participants_show')]
    public function show(string $uuid, ParticipantRepositoryInterface $repository, #[MapQueryParameter(name: 'kommentare')] ?bool $commentsOnly, Breadcrumb $breadcrumb): Response
    {
        if (!($entity = $repository->findByUuid($uuid)) instanceof ParticipantInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('view', $entity);

        return $this->render('@FerienpassAdmin/page/participants/show.html.twig', [
            'item' => $entity,
            'commentsOnly' => (bool) $commentsOnly,
            'breadcrumb' => $breadcrumb->generate(['participants.title', ['route' => 'admin_participants_index']], $entity->getName()),
        ]);
    }

    #[Route('/{uuid}/anmeldungen', name: 'admin_participants_attendances', requirements: ['uuid' => Requirement::UUID])]
    public function attendances(string $uuid, ParticipantRepositoryInterface $participantRepository, AttendanceRepository $attendanceRepository, Request $request, Breadcrumb $breadcrumb): Response
    {
        /** @var ParticipantInterface $participant */
        $participant = $participantRepository->findByUuid($uuid);
        if (null === $participant) {
            throw $this->createNotFoundException();
        }

        $qb = $attendanceRepository->createQueryBuilder('i');
        $qb
            ->leftJoin('i.offer', 'o')
            ->leftJoin('o.dates', 'd')
            ->innerJoin('i.participant', 'p')
            ->where('p.id = :participant')
            ->setParameter('participant', $participant->getId())
        ;

        /** @var Attendance[] $items */
        $items = $qb->getQuery()->getResult();

        $ms = new MultiSelect(['settle'], self::class, array_values(array_map(fn (Attendance $a) => $a->getId(), array_filter($items, fn (Attendance $a) => $a->isConfirmed() && !$a->isPaid() && (!$a->getOffer()->getEdition()->isCompleted())))));

        return $this->render('@FerienpassAdmin/page/participants/attendances.html.twig', [
            'qb' => $qb,
            'filterType' => AttendancesFilter::class,
            'ms' => $ms,
            'searchable' => ['o.name'],
            'participant' => $participant,
            'breadcrumb' => $breadcrumb->generate(['participants.title', ['route' => 'admin_participants_index']], $participant->getName().' (Anmeldungen)'),
        ]);
    }

    public function handleMultiSelect(string $action, array $selected, Request $request): Response
    {
        if ('settle' === $action) {
            $request->getSession()->set('ms.settle', array_values($selected));

            return $this->redirectToRoute('admin_attendances_settle');
        }

        throw new \RuntimeException('Code should not be reached');
    }

    #[Route('/{id}/löschen', name: 'admin_participants_delete', requirements: ['id' => '\d+'])]
    public function delete(int $id, ParticipantRepositoryInterface $repository, Request $request, EntityManagerInterface $entityManager): Response
    {
        /** @var ParticipantInterface $item */
        if (null === $item = $repository->find($id)) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('delete', $item);

        $form = $this->createForm(FormType::class, options: [
            'action' => $this->generateUrl('admin_participants_delete', ['id' => $id]),
        ]);
        $form->handleRequest($request);

        // TODO Can use a facade
        if ($form->isSubmitted() && $form->isValid()) {
            foreach ($item->getAttendances() as $attendance) {
                foreach ($attendance->getPaymentItems() as $paymentItem) {
                    $paymentItem->removeAttendanceAssociation();
                }
            }

            $deletedId = $item->getId();

            $entityManager->remove($item);
            $entityManager->flush();

            if (TurboBundle::STREAM_FORMAT === $request->getPreferredFormat()) {
                // If the request comes from Turbo, set the content type as text/vnd.turbo-stream.html and only send the HTML to update
                $request->setRequestFormat(TurboBundle::STREAM_FORMAT);

                return $this->renderBlock('@FerienpassAdmin/page/participants/index.html.twig', 'deleted_stream', ['id' => $deletedId]);
            }

            return $this->redirectToRoute('admin_participants_index');
        }

        return $this->render('@FerienpassAdmin/page/participants/delete.html.twig', [
            'item' => $item,
            'form' => $form,
        ]);
    }

    #[Route('/anmeldung/{uuid:attendance}/zahllast', name: 'admin_attendance_pay_status', requirements: ['participant' => Requirement::UUID, 'attendance' => Requirement::UUID])]
    public function attendance(Attendance $attendance, Request $request, EntityManagerInterface $entityManager, \Psr\EventDispatcher\EventDispatcherInterface $dispatcher): Response
    {
        $form = $this->createForm(FormType::class, options: [
            'action' => $this->generateUrl('admin_attendance_pay_status', [
                'uuid' => $attendance->getUuid(),
            ]),
        ]);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $attendance->setPayable(!$attendance->isPayable());

            $entityManager->flush();

            return $this->redirectToRoute('admin_attendance_pay_status', [
                'uuid' => $attendance->getUuid(),
            ]);
        }

        return $this->render('@FerienpassAdmin/page/attendances/pay_status.html.twig', [
            'item' => $attendance,
            'form' => $form,
            'dispatcher' => $dispatcher,
        ]);
    }

    #[Route('/abrechnen', name: 'admin_attendances_settle')]
    public function settle(Request $request, Breadcrumb $breadcrumb, AttendanceRepository $attendanceRepository, EventDispatcherInterface $dispatcher, WorkflowInterface $paymentStateMachine): Response
    {
        $user = $this->getUser();
        $attendances = $this->getAttendancesFromRequest($attendanceRepository, $request);
        $attendances = array_filter($attendances, fn (Attendance $a) => !$a->isPaid());

        $possibleDebtors = array_unique(array_filter(array_map(fn (Attendance $a) => $a->getParticipant()->getUser()?->getDebtor(), $attendances)));
        if (1 !== \count($possibleDebtors)) {
            // TODO: Make user select a debtor
        }

        /** @var Debtor $debtor */
        $debtor = $possibleDebtors[0] ?? new Debtor();
        $draftPayment = Payment::fromAttendances($debtor, $attendances, $dispatcher, user: $user);

        if ($debtor->getBalance()) {
            $draftPayment->addItem(new PaymentItem($draftPayment, (-1) * $debtor->getBalance(), type: 'discount'));
        }

        $form = $this->createForm(SettleAttendancesType::class, $dto = BillingAddressDto::fromPayment($draftPayment), ['attendances' => $attendances]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $payment = new Payment($debtor, user: $user);
            $dto->toPayment($payment);

            if ($dto->isPaid()) {
                $paymentStateMachine->apply($payment, Payment::TRANSITION_PAY);
            }

            $em = $this->doctrine->getManager();
            $em->persist($payment);
            $em->flush();

            return $this->redirectToRoute('admin_payments_receipt', ['id' => $payment->getId()]);
        }

        /** @noinspection FormViewTemplate `createView()` messes ups error handling/redirect */
        return $this->render('@FerienpassAdmin/page/participants/settle.html.twig', [
            'form' => $form,
            'payment' => $draftPayment,
            'breadcrumb' => $breadcrumb->generate(['participants.title', ['route' => 'admin_participants_index']], 'Anmeldungen abrechnen'),
        ]);
    }

    /**
     * @return array<Attendance>
     */
    private function getAttendancesFromRequest(AttendanceRepository $attendances, Request $request): array
    {
        $session = $request->getSession();
        if ($session->has('ms.settle')) {
            $ids = $session->get('ms.settle');
        }

        if ($request->request->has(SettleAttendancesType::FORM_NAME)) {
            $ids = $request->get(SettleAttendancesType::FORM_NAME)['ms'] ?? [];
        }

        if (null === ($ids ?? null)) {
            return [];
        }

        return $attendances->findBy(['id' => $ids]);
    }
}
