<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\QueryBuilder;
use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\AdminBundle\Export\ExportQueryBuilderInterface;
use Ferienpass\AdminBundle\Form\Filter\OffersFilter;
use Ferienpass\AdminBundle\WorkflowSurvey\OfferCompleteSurveyInterface;
use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Entity\Notification;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Offer\VariantsTrait;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Entity\WorkflowSurvey;
use Ferienpass\CoreBundle\Export\Offer\OffersExportInterface;
use Ferienpass\CoreBundle\Message\SyncOfferVariants;
use Ferienpass\CoreBundle\Repository\EditionRepository;
use Ferienpass\CoreBundle\Repository\HostRepository;
use Ferienpass\CoreBundle\Repository\NotificationRepository;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Ferienpass\CoreBundle\Repository\WorkflowSurveyRepository;
use Knp\Menu\FactoryInterface;
use Symfony\Bridge\Doctrine\Attribute\MapEntity;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Attribute\TaggedLocator;
use Symfony\Component\DependencyInjection\ServiceLocator;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Attribute\MapQueryParameter;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Requirement\Requirement;
use Symfony\Component\Security\Http\Attribute\CurrentUser;
use Symfony\Component\Uid\Uuid;
use Symfony\Component\Workflow\WorkflowInterface;
use Symfony\UX\Turbo\TurboBundle;

#[Route('/angebote')]
final class OffersController extends AbstractController
{
    public function __construct(#[TaggedLocator(ExportQueryBuilderInterface::class, defaultIndexMethod: 'getFormat')] private readonly ServiceLocator $exporters, #[Autowire(param: 'ferienpass.model.offer.class')] private readonly string $offerEntityClass, private readonly MessageBusInterface $commandBus)
    {
    }

    #[Route('/{edition?}', name: 'admin_offers_index', priority: -2)]
    public function index(#[MapEntity(mapping: ['edition' => 'alias'])] ?Edition $edition, #[CurrentUser] User $user, OfferRepositoryInterface $repository, HostRepository $hosts, Breadcrumb $breadcrumb, FactoryInterface $factory, EditionRepository $editions, Request $request): Response
    {
        if (null !== $request->attributes->get('edition') && !$edition instanceof Edition) {
            throw $this->createNotFoundException();
        }

        $qb = $repository->createQueryBuilder('i')
            ->leftJoin('i.dates', 'd')
        ;

        // Solve N+1
        $qb
            ->addSelect('d')
            ->leftJoin('i.hosts', 'h')
            ->addSelect('h')
            ->leftJoin('i.variants', 'v')
            ->addSelect('v')
            ->leftJoin('i.activity', 'l')
            ->addSelect('l')
            ->leftJoin('i.attendances', 'a')
            ->addSelect('a')
        ;

        $menu = $factory->createItem('offers.editions');

        foreach ($editions->findBy(['archived' => false], ['createdAt' => 'DESC']) as $e) {
            if (!$this->isGranted('ROLE_ADMIN') && !$e->getHosts()->isEmpty() && !$e->getHosts()->contains($user->getHosts()->first())) {
                continue;
            }

            $menu->addChild($e->getName(), [
                'route' => 'admin_offers_index',
                'routeParameters' => ['edition' => $e->getAlias()],
                'current' => $edition instanceof Edition && $e->getAlias() === $edition->getAlias(),
            ]);
        }

        $draft = OfferInterface::STATE_DRAFT;

        $unfinalizedOffers = $repository->createQueryBuilder('i')
            ->select('COUNT(i)')
            ->innerJoin('i.hosts', 'h')
            ->andWhere("JSON_CONTAINS_PATH(i.status, 'one', '$.$draft') = 1")
            ->andWhere('i.edition = :edition')->setParameter('edition', $edition)
            ->andWhere('h IN (:hosts)')->setParameter('hosts', $hosts->findByUser($user))
            ->getQuery()
            ->getSingleScalarResult()
        ;

        return $this->render('@FerienpassAdmin/page/offers/index.html.twig', [
            'qb' => $qb,
            'filterType' => OffersFilter::class,
            'createUrl' => $edition instanceof Edition ? ((!$edition instanceof Edition && (!$menu->hasChildren() || $this->isGranted('ROLE_ADMIN'))) || $this->isGranted('offer.create', $edition) ? $this->generateUrl('admin_offers_new_in_edition', array_filter(['edition' => $edition->getAlias()])) : null) : ($this->generateUrl('admin_offers_create')),
            'exports' => $this->isGranted('ROLE_ADMIN') ? array_keys($this->exporters->getProvidedServices()) : [],
            'searchable' => ['name'],
            'edition' => $edition,
            'unfinalizedOffers' => $unfinalizedOffers > 0,
            'aside_nav' => $menu,
            'breadcrumb' => $breadcrumb->generate(['offers.title', ['route' => 'admin_offers_index']], $edition?->getName()),
        ]);
    }

    #[Route('/export.{format}', name: 'admin_offers_export', requirements: ['format' => '\w+'])]
    public function export(OfferRepositoryInterface $repository, string $format)
    {
        $qb = $repository->createQueryBuilder('i');
        $qb->orderBy('i.createdAt', 'DESC');

        if (!$this->exporters->has($format)) {
            throw $this->createNotFoundException();
        }

        $exporter = $this->exporters->get($format);

        return $this->file($exporter->generate($qb), "angebote.$format");
    }

    #[Route('/{uuid}/info', name: 'admin_offers_show')]
    public function show(string $uuid, #[MapQueryParameter(name: 'kommentare')] ?bool $commentsOnly, OfferRepositoryInterface $repository, Breadcrumb $breadcrumb): Response
    {
        if (null === $entity = $repository->findOneBy(['uuid' => $uuid])) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('view', $entity);

        return $this->render('@FerienpassAdmin/page/offers/show.html.twig', [
            'item' => $entity,
            'commentsOnly' => (bool) $commentsOnly,
            'breadcrumb' => $breadcrumb->generate(
                ['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => array_filter(['edition' => $entity->getEdition()?->getAlias()])]],
                $entity->getEdition() ? [$entity->getEdition()->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $entity->getEdition()->getAlias()]]] : null,
                $entity->getName()
            ),
        ]);
    }

    #[Route('/neu', name: 'admin_offers_new')]
    #[Route('/{edition}/neu', name: 'admin_offers_new_in_edition')]
    public function newWizard(#[MapEntity(mapping: ['edition' => 'alias'])] ?Edition $edition, Breadcrumb $breadcrumb, Request $request): Response
    {
        if ($edition instanceof Edition) {
            $this->denyAccessUnlessGranted('offer.create', $edition);
        }

        if ('admin_offers_new_in_edition' === $request->attributes->get('_route') && !$edition instanceof Edition) {
            throw $this->createNotFoundException();
        }

        return $this->render('@FerienpassAdmin/page/offers/new.html.twig', [
            'edition' => $edition,
            'breadcrumb' => $breadcrumb->generate(
                ['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => array_filter(['edition' => $edition?->getAlias()])]],
                $edition instanceof Edition ? [$edition->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $edition->getAlias()]]] : null,
                'Neue Veranstaltung'
            ),
        ]);
    }

    #[Route('/{uuid}/korrekturabzug', name: 'admin_offers_proof')]
    public function proof(string $uuid, OfferRepositoryInterface $repository, #[TaggedLocator(OffersExportInterface::class, indexAttribute: 'name', defaultIndexMethod: 'getName')] ServiceLocator $exporters, Request $request, WorkflowInterface $offerWorkflow, Breadcrumb $breadcrumb, EntityManagerInterface $entityManager): Response
    {
        if (!($offer = $repository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('view', $offer);

        if ($request->isMethod('POST')) {
            $offerWorkflow->apply($offer, OfferInterface::TRANSITION_FINALIZE);

            $entityManager->flush();

            return $this->redirectToRoute('admin_offers_proof', ['uuid' => $offer->getUuid()]);
        }

        return $this->render('@FerienpassAdmin/page/offers/proof.html.twig', [
            'offer' => $offer,
            'pdfs' => array_map(fn ($v) => substr($v, 0, -4), array_filter(array_keys($exporters->getProvidedServices()), fn (string $k) => str_ends_with($k, '.pdf'))),
            'breadcrumb' => $breadcrumb->generate(
                ...array_filter([
                    ['offers.title', ['route' => 'admin_offers_index']],
                    [$offer->getEdition()->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]],
                    $offer->getName(),
                ])
            ),
        ]);
    }

    #[Route('/{uuid}/abschließen', name: 'admin_offers_complete')]
    public function complete(string $uuid, OfferRepositoryInterface $repository, WorkflowSurveyRepository $surveyData, Request $request, Breadcrumb $breadcrumb, #[TaggedLocator(OfferCompleteSurveyInterface::class, defaultIndexMethod: 'getName')] ServiceLocator $surveys, EntityManagerInterface $entityManager, WorkflowInterface $offerWorkflow): Response
    {
        /** @var OfferInterface $offer */
        if (!($offer = $repository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        if (!$offerWorkflow->can($offer, OfferInterface::TRANSITION_COMPLETE)) {
            throw $this->createAccessDeniedException();
        }

        $edition = $offer->getEdition();
        $key = $offer->isOnlineApplication() ? $edition->getSurveyWithApplication() : $edition->getSurveyWithoutApplication();
        if (null === $key) {
            throw $this->createNotFoundException();
        }

        if (!$surveys->has($key)) {
            throw $this->createNotFoundException();
        }

        /** @var OfferCompleteSurveyInterface $survey */
        $survey = $surveys->get($key);

        /** @var WorkflowSurvey|null $data */
        $data = $surveyData->findOneBy(['offer' => $offer, 'survey' => $key]);

        // TODO real DTO
        $dto = $data?->getData() ?? [];
        $form = $this->createForm($survey::class, $dto);
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $data ??= new WorkflowSurvey($offer, $key);
            $data->setData($form->getData());

            if (!$data->getId()) {
                $entityManager->persist($data);
            }

            $offerWorkflow->apply($offer, OfferInterface::TRANSITION_COMPLETE, ['survey_data' => $data->getData()]);
            $entityManager->flush();

            return $this->redirectToRoute('admin_offers_index', ['edition' => $offer->getEdition()->getAlias()]);
        }

        return $this->render('@FerienpassAdmin/page/offers/complete.html.twig', [
            'offer' => $offer,
            'action' => $survey,
            'form' => $form,
            'breadcrumb' => $breadcrumb->generate(['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]], [$offer->getEdition()->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]], $offer->getName()),
        ]);
    }

    #[Route('/{uuid}/neu', name: 'admin_offers_duplicate', requirements: ['uuid' => Requirement::UUID], priority: 1)]
    public function duplicate(string $uuid, OfferRepositoryInterface $repository, EditionRepository $editionRepository): Response
    {
        if (!($entity = $repository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('view', $entity);

        $editions = $editionRepository->findWithActiveTask('host_editing_stage');

        return $this->render('@FerienpassAdmin/page/offers/duplicate.html.twig', [
            'item' => $entity,
            'editions' => $editions,
        ]);
    }

    #[Route('/{uuid}/varianten', name: 'admin_offers_variants', requirements: ['uuid' => Requirement::UUID])]
    public function variants(string $uuid, OfferRepositoryInterface $repository, Request $request, EntityManagerInterface $entityManager, #[CurrentUser] User $user, FormFactoryInterface $formFactory): Response
    {
        /** @var OfferInterface $item */
        if (!($item = $repository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('edit', $item);

        /** @var VariantsTrait $item */
        if ($item->isVariant() || $item->hasVariants()) {
            $detach = $formFactory->createNamed('detach', options: [
                'action' => $this->generateUrl('admin_offers_variants', ['uuid' => $uuid]),
            ]);

            $detach->handleRequest($request);
            if ($detach->isSubmitted() && $detach->isValid()) {
                if ($item->hasVariants()) {
                    $variants = $item->getVariants()->toArray();
                    $variants[0]->setVariantBase(null);
                    $counter = \count($variants);

                    for ($i = 1; $i < $counter; ++$i) {
                        $variants[$i]->setVariantBase($variants[0]);
                    }
                } elseif (!$item->isVariantBase()) {
                    $item->setVariantBase(null);
                }

                $entityManager->flush();

                return $this->redirectToRoute('admin_offers_variants', ['uuid' => $uuid]);
            }
        } else {
            $link = $formFactory->createNamedBuilder('link', options: [
                'action' => $this->generateUrl('admin_offers_variants', ['uuid' => $uuid]),
            ])
                ->add('base', EntityType::class, [
                    'class' => $this->offerEntityClass,
                    'query_builder' => function (EntityRepository $er) use ($item, $user): QueryBuilder {
                        $qb = $er->createQueryBuilder('o')
                            ->andWhere('o.edition = :edition')
                            ->setParameter('edition', $item->getEdition())
                            ->andwhere('o.variantBase IS NULL')
                            ->andWhere('o.id <> :id')
                            ->setParameter('id', $item->getId());

                        if (!$this->isGranted('ROLE_ADMIN')) {
                            $qb->innerJoin('o.hosts', 'h')
                                ->andWhere('h IN (:hosts)')
                                ->setParameter('hosts', $user->getHosts())
                            ;
                        }

                        return $qb->orderBy('o.name', 'ASC');
                    },
                    'choice_label' => 'name',
                    'label' => 'Haupttermin auswählen',
                    'placeholder' => '-',
                ])
                ->getForm()
            ;

            $link->handleRequest($request);
            if ($link->isSubmitted() && $link->isValid()) {
                $base = $link->get('base')->getData();

                $item->setVariantBase($base);

                $entityManager->flush();

                $this->commandBus->dispatch(new SyncOfferVariants($base->getId()));

                return $this->redirectToRoute('admin_offers_variants', ['uuid' => $uuid]);
            }
        }

        return $this->render('@FerienpassAdmin/page/offers/variants.html.twig', [
            'item' => $item,
            'variants' => $item->getVariants(true),
            'detach' => $detach ?? null,
            'link' => $link ?? null,
        ]);
    }

    #[Route('/{uuid}/löschen', name: 'admin_offers_delete')]
    public function delete(string $uuid, OfferRepositoryInterface $repository, Request $request, EntityManagerInterface $entityManager): Response
    {
        if (!($item = $repository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('delete', $item);

        $form = $this->createForm(FormType::class, options: [
            'action' => $this->generateUrl('admin_offers_delete', ['uuid' => $uuid]),
        ]);
        $form->handleRequest($request);

        // TODO Can use a facade
        if ($form->isSubmitted() && $form->isValid()) {
            /** @var $attendance */
            foreach ($item->getAttendances() as $attendance) {
                foreach ($attendance->getPaymentItems() as $paymentItem) {
                    $paymentItem->removeAttendanceAssociation();
                }
            }

            if ($item->hasVariants()) {
                $variants = $item->getVariants()->toArray();
                $variants[0]->setVariantBase(null);
                $counter = \count($variants);

                for ($i = 1; $i < $counter; ++$i) {
                    $variants[$i]->setVariantBase($variants[0]);
                }
            } elseif (!$item->isVariantBase()) {
                $item->setVariantBase(null);
            }

            $deletedId = $item->getId();
            $edition = $item->getEdition()->getAlias();

            $entityManager->remove($item);
            $entityManager->flush();

            if (TurboBundle::STREAM_FORMAT === $request->getPreferredFormat()) {
                // If the request comes from Turbo, set the content type as text/vnd.turbo-stream.html and only send the HTML to update
                $request->setRequestFormat(TurboBundle::STREAM_FORMAT);

                return $this->renderBlock('@FerienpassAdmin/page/offers/index.html.twig', 'deleted_stream', ['id' => $deletedId]);
            }

            return $this->redirectToRoute('admin_offers_index', ['edition' => $edition]);
        }

        return $this->render('@FerienpassAdmin/page/offers/delete.html.twig', [
            'item' => $item,
            'form' => $form,
        ]);
    }

    #[Route('/{uuid}/absagen', name: 'admin_offers_cancel', requirements: ['uuid' => Requirement::UUID])]
    public function cancel(string $uuid, OfferRepositoryInterface $repository, Request $request, EntityManagerInterface $entityManager, WorkflowInterface $offerWorkflow, Breadcrumb $breadcrumb, NotificationRepository $notificationRepository): Response
    {
        if (!($offer = $repository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted(OfferInterface::TRANSITION_CANCEL, $offer);

        $form = $this->createFormBuilder(options: [
            'action' => $this->generateUrl('admin_offers_cancel', ['uuid' => $uuid]),
        ])
            ->add('notifyText', TextareaType::class, [
                'label' => 'E-Mail-Text',
                'required' => false,
                'data' => $this->notificationText('offer_cancelled', $offer, $notificationRepository),
            ])
            ->getForm()
        ;
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $transitionUuid = Uuid::v4();

            $offerWorkflow->apply($offer, OfferInterface::TRANSITION_CANCEL, [
                'uuid' => (string) $transitionUuid,
                'notifyText' => $form->get('notifyText')->getData(),
            ]);

            $entityManager->flush();

            if ($offer->isOnlineApplication()) {
                return $this->redirectToRoute('admin_tools_dispatch', ['uuid' => (string) $transitionUuid]);
            }

            return $this->redirectToRoute('admin_offers_proof', ['uuid' => $offer->getUuid()]);
        }

        return $this->render('@FerienpassAdmin/page/offers/cancel.html.twig', [
            'offer' => $offer,
            'cancel' => $form,
            'breadcrumb' => $breadcrumb->generate(['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]], [$offer->getEdition()->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]], $offer->getName()),
        ]);
    }

    #[Route('/{uuid}/wiederherstellen', name: 'admin_offers_relaunch', requirements: ['uuid' => Requirement::UUID])]
    public function relaunch(string $uuid, OfferRepositoryInterface $repository, Request $request, EntityManagerInterface $entityManager, WorkflowInterface $offerWorkflow, Breadcrumb $breadcrumb, NotificationRepository $notificationRepository): Response
    {
        /** @var OfferInterface $offer */
        if (!($offer = $repository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted(OfferInterface::TRANSITION_RELAUNCH, $offer);

        $form = $this->createFormBuilder(options: [
            'action' => $this->generateUrl('admin_offers_relaunch', ['uuid' => $uuid]),
        ])
            ->add('notifyText', TextareaType::class, [
                'label' => 'E-Mail-Text',
                'required' => false,
                'data' => $this->notificationText('offer_relaunched', $offer, $notificationRepository),
            ])
            ->getForm()
        ;
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $transitionUuid = Uuid::v4();

            $offerWorkflow->apply($offer, OfferInterface::TRANSITION_RELAUNCH, [
                'uuid' => (string) $transitionUuid,
                'notifyText' => $form->get('notifyText')->getData(),
            ]);

            $entityManager->flush();

            if ($offer->isOnlineApplication()) {
                return $this->redirectToRoute('admin_tools_dispatch', ['uuid' => (string) $transitionUuid]);
            }

            return $this->redirectToRoute('admin_offers_proof', ['uuid' => $uuid]);
        }

        return $this->render('@FerienpassAdmin/page/offers/relaunch.html.twig', [
            'offer' => $offer,
            'relaunch' => $form,
            'breadcrumb' => $breadcrumb->generate(
                ['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]],
                [$offer->getEdition()->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]],
                $offer->getName()
            ),
        ]);
    }

    private function notificationText(string $notification, OfferInterface $offer, NotificationRepository $notificationRepository): ?string
    {
        $entity = $notificationRepository
            ->createQueryBuilder('n')
            ->where('n.type = :type')
            ->andWhere('n.disable = 0')
            ->setParameter('type', $notification)
        ;

        $entity
            ->addSelect('(CASE WHEN n.edition = :edition THEN 1 ELSE 0 END) AS HIDDEN mainSort')
            ->addOrderBy('mainSort', 'DESC')
            ->setParameter('edition', $offer->getEdition())
        ;

        $entity->addSelect('(CASE WHEN n.edition IS NULL THEN 1 ELSE 0 END) AS HIDDEN defaultFirst')->addOrderBy('defaultFirst', 'DESC');

        $entity = $entity->setMaxResults(1)->getQuery()->getOneOrNullResult();

        if (!($entity instanceof Notification)) {
            return null;
        }

        return $entity->getEmailText();
    }
}
