<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\CmsBundle\Form\SimpleType\ContaoRequestTokenType;
use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Message\LotEdition;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Lock\LockFactory;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\CurrentUser;
use Symfony\Component\Security\Http\Attribute\IsGranted;

#[IsGranted('ROLE_ADMIN')]
class OffersAssignController extends AbstractController
{
    #[Route('/angebote/{alias}/zuordnen', name: 'admin_editions_assign')]
    #[Route('/angebote/zuordnen', name: 'admin_offers_assign', priority: 2)]
    public function __invoke(?Edition $edition, #[CurrentUser] User $user, LockFactory $lockFactory, Request $request, Breadcrumb $breadcrumb, MessageBusInterface $bus): Response
    {
        $lock = $lockFactory->createLock('lotting.'.$edition?->getId());

        /** @noinspection FormViewTemplate `createView()` messes ups error handling/redirect */
        return $this->render('@FerienpassAdmin/page/offers/assignAll.html.twig', [
            'locked' => !$lock->acquire(),
            'edition' => $edition,
            'breadcrumb' => $breadcrumb->generate(['offers.title', ['route' => 'admin_offers_index']], [$edition?->getName() ?? '', ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $edition->getAlias()]]], 'Plätze zuteilen'),
        ]);
    }

    #[Route('/saisons/{alias}/zuordnen/auto', name: 'admin_editions_lotting')]
    public function dispatchLotting(Edition $edition, Request $request, MessageBusInterface $commandBus): Response
    {
        $this->denyAccessUnlessGranted('lotting', $edition);

        $form = $this->createFormBuilder(options: [
            'action' => $this->generateUrl('admin_editions_lotting', ['alias' => $edition->getAlias()]),
        ]);

        $form->add('requestToken', ContaoRequestTokenType::class);
        $form->add('waitlist', ChoiceType::class, [
            'label' => 'Überhang-Teilnehmende sollen…',
            'choices' => [
                '…auf die Warteliste' => 'waitlist',
                '…abgelehnt werden' => 'reject',
            ]])
        ;
        $form->add('ignoreMinParticipants', CheckboxType::class, [
            'label' => 'Auch Angebote zulosen, die die Mindestteilnahmezahl nicht erreichen',
            'required' => false,
        ]);

        $form = $form->getForm();
        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $commandBus->dispatch(new LotEdition($edition->getId(), 'waitlist' === $form->get('waitlist')->getData(), (bool) $form->get('ignoreMinParticipants')->getData()));

            return $this->redirectToRoute('admin_editions_assign', ['alias' => $edition->getAlias()]);
        }

        return $this->render('@FerienpassAdmin/page/edition/lotting.html.twig', [
            'edition' => $edition,
            'form' => $form,
        ]);
    }
}
