<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\AdminBundle\ApplicationSystem\ParticipantListDto;
use Ferienpass\AdminBundle\ApplicationSystem\ParticipantListFacade;
use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\AdminBundle\LiveComponent\MultiSelect;
use Ferienpass\AdminBundle\LiveComponent\MultiSelectHandlerInterface;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\HostConsent;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Export\Offer\ConsentsOverviewExport;
use Ferienpass\CoreBundle\Facade\AttendanceFacade;
use Ferienpass\CoreBundle\Repository\AttendanceRepository;
use Ferienpass\CoreBundle\Repository\HostConsentRepository;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Ferienpass\CoreBundle\Session\Flash;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

#[Route('/angebote/{uuid}')]
final class OfferParticipantsController extends AbstractController implements MultiSelectHandlerInterface
{
    public function __construct(private readonly ParticipantListFacade $participantList, private readonly HostConsentRepository $consents, private readonly AttendanceRepository $attendances, private readonly Flash $flash, #[Autowire(param: 'ferienpass_admin.privacy_consent_text')] private readonly string $consentText, #[Autowire(param: 'ferienpass_admin.participant_list.hosts_can_edit')] private readonly bool $hostsCanEdit)
    {
    }

    #[Route('/teilnahmeliste', name: 'admin_offers_participants')]
    public function __invoke(string $uuid, OfferRepositoryInterface $offers, AttendanceRepository $attendanceRepository, Request $request, EntityManagerInterface $em, AttendanceFacade $attendanceFacade, Breadcrumb $breadcrumb, Flash $flash, ParticipantListDto $participantList): Response
    {
        /** @var OfferInterface $offer */
        if (!($offer = $offers->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('participants.view', $offer);

        $qb = $attendanceRepository->createQueryBuilder('i');
        $qb
            ->innerJoin('i.participant', 'p')
            ->innerJoin('i.offer', 'o')
            ->where('o.id = :offer')
            ->setParameter('offer', $offer->getId())
        ;

        if (!$this->isGranted('ROLE_ADMIN')) {
            $qb->andWhere('i.status <> :status_withdrawn')->setParameter('status_withdrawn', Attendance::STATUS_WITHDRAWN);

            if (!$this->hostsCanEdit) {
                $qb->andWhere('i.status = :status_confirmed')->setParameter('status_confirmed', Attendance::STATUS_CONFIRMED);
            }
        }

        $user = $this->getUser();
        if (!$user instanceof User) {
            return new Response('', Response::HTTP_NO_CONTENT);
        }

        if ('' !== $this->consentText && !$this->consents->findValid($user) instanceof HostConsent) {
            return $this->render('@FerienpassAdmin/page/missing_privacy_statement.html.twig', [
                'breadcrumb' => $breadcrumb->generate(
                    ['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()?->getAlias()]]],
                    [$offer->getEdition()?->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()?->getAlias()]]],
                    $offer->getName(),
                    'Anmeldungen'
                ),
            ]);
        }

        $ms = null;
        if ($this->isGranted('participants.edit', $offer)) {
            $ms = new MultiSelect(['confirm', 'reject'], self::class);
        }

        return $this->render('@FerienpassAdmin/page/offers/participant_list.html.twig', [
            'qb' => $qb,
            'searchable' => ['p.firstname', 'p.lastname'],
            'ms' => $ms,
            'offer' => $offer,
            'columns' => $participantList->adminViewColumns($offer),
            'breadcrumb' => $breadcrumb->generate(
                ['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => array_filter(['edition' => $offer->getEdition()?->getAlias()])]],
                [$offer->getEdition()?->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()?->getAlias()]]],
                $offer->getName(),
                'Anmeldungen'
            ),
        ]);
    }

    public function handleMultiSelect(string $action, array $selected, Request $request): Response
    {
        /** @var Attendance[] $selectedParticipants */
        $selectedParticipants = $this->attendances->findBy(['id' => $selected]);
        if (empty($selectedParticipants)) {
            return new Response('', Response::HTTP_BAD_REQUEST);
        }

        $offer = $selectedParticipants[0]->getOffer();

        $this->denyAccessUnlessGranted("participants.$action", $offer);

        if ('confirm' === $action) {
            $this->participantList->confirm($selectedParticipants);

            $this->flash->addConfirmation(text: 'Den Teilnehmer:innen wurde zugesagt.');
        }

        if ('reject' === $action) {
            $this->participantList->reject($selectedParticipants);

            $this->flash->addConfirmation(text: 'Den Teilnehmer:innen wurde abgesagt.');
        }

        return $this->redirectToRoute('admin_offers_participants', ['uuid' => $offer->getUuid()]);
    }

    #[Route('/einwilligungen', name: 'admin_offer_consents_pdf')]
    public function consents(string $uuid, OfferRepositoryInterface $offers, ConsentsOverviewExport $export): Response
    {
        /** @var OfferInterface $offer */
        if (!($offer = $offers->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('participants.view', $offer);

        $path = $export->generate($offer, true);

        return $this->file($path, \sprintf('einwilligungen-%s.pdf', $offer->getAlias() ?: (string) $offer->getId()));
    }
}
