<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Contao\CoreBundle\Exception\PageNotFoundException;
use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\AdminBundle\Form\EditOfferType;
use Ferienpass\AdminBundle\Form\FormScreenFactory;
use Ferienpass\AdminBundle\Form\HandleFormTrait;
use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\OfferDate;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Symfony\Bridge\Doctrine\Attribute\MapEntity;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Requirement\Requirement;

#[Route('/angebote')]
final class OfferEditController extends AbstractController
{
    use HandleFormTrait;

    public function __construct(private readonly OfferRepositoryInterface $offers, private readonly EntityManagerInterface $entityManager)
    {
    }

    #[Route('/{uuid}/{section?default}', name: 'admin_offers_edit', requirements: ['uuid' => Requirement::UUID], priority: -2)]
    #[Route('/neu/leer/{section?default}', name: 'admin_offers_create')]
    #[Route('/{edition}/neu/leer/{section?default}', name: 'admin_offers_create_in_edition')]
    #[Route('/{uuid}/kopieren/{section?default}', name: 'admin_offers_copy', requirements: ['uuid' => Requirement::UUID], priority: -1)]
    #[Route('/{edition}/{uuid}/kopieren/{section?default}', name: 'admin_offers_copy_to_edition', requirements: ['uuid' => Requirement::UUID], priority: -1)]
    #[Route('/{uuid}/variante/{section?default}', name: 'admin_offers_new_variant', requirements: ['uuid' => Requirement::UUID], priority: -1)]
    public function __invoke(?string $uuid, #[MapEntity(mapping: ['edition' => 'alias'])] ?Edition $edition, string $section, Request $request, Breadcrumb $breadcrumb, FormScreenFactory $forms, EventDispatcherInterface $dispatcher, EntityManagerInterface $entityManager): Response
    {
        $offer = $this->getOffer($uuid, $edition, $request);

        $formScreen = $forms->screen(EditOfferType::class, $offer);
        if (!\in_array($section, $formScreen->sections(), true)) {
            throw new PageNotFoundException('Section not found');
        }

        $form = $this->createForm(EditOfferType::class, $offer, $formOptions = [
            'is_variant' => !$offer->isVariantBase(),
            'display_section' => $section,
            'disabled' => $isReadonly = $offer->getId() && !$this->isGranted('edit', $offer),
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $this->handleSubmit($form, $dispatcher, $request);

            $entityManager->flush();

            return $this->redirectToRoute('admin_offers_edit', ['uuid' => $offer->getUuid(), 'section' => $section]);
        }

        return $this->render('@FerienpassAdmin/page/offers/edit.html.twig', [
            'item' => $offer,
            'form' => $form,
            'section' => $section,
            'isReadonly' => $isReadonly,
            'formOptions' => $formOptions,
            'formScreen' => $formScreen,
            'breadcrumb' => $breadcrumb->generate(
                ['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => array_filter(['edition' => $offer->getEdition()?->getAlias()])]],
                [$offer->getEdition()?->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()?->getAlias()]]],
                $offer->getName().' (bearbeiten)'
            ),
        ]);
    }

    private function getOffer(?string $uuid, ?Edition $edition, Request $request): OfferInterface
    {
        $user = $this->getUser();

        $offer = null;
        if (null !== $uuid) {
            /** @var OfferInterface $offer */
            $offer = $this->offers->findOneBy(['uuid' => $uuid]);
        }

        if ('admin_offers_edit' === $request->get('_route') && null === $offer) {
            throw new PageNotFoundException('Item not found');
        }

        if (null === $offer) {
            $offer = $this->offers->createNew();
            if ($edition instanceof Edition) {
                $offer->setEdition($edition);
            }

            $offer->addDate(new OfferDate($offer));

            if ($user instanceof User) {
                $offer->setContact([[
                    'name' => $user->getName(),
                    'email' => $user->getEmail(),
                    'phone' => $user->getPhone(),
                ]]);
            }

            $this->denyAccessUnlessGranted('create', $offer);

            $this->entityManager->persist($offer);

            return $offer;
        }

        if ('admin_offers_copy' === $request->get('_route')) {
            $copy = $this->offers->createCopy($offer);

            $this->denyAccessUnlessGranted('view', $offer);
            $this->denyAccessUnlessGranted('create', $copy);

            $this->entityManager->persist($copy);

            return $copy;
        }

        if ('admin_offers_copy_to_edition' === $request->get('_route')) {
            $copy = $this->offers->createCopy($offer);

            if (!$edition instanceof Edition) {
                throw new PageNotFoundException('Edition not found');
            }

            $copy->setEdition($edition);

            $this->denyAccessUnlessGranted('view', $offer);
            $this->denyAccessUnlessGranted('create', $copy);

            $this->entityManager->persist($copy);

            return $copy;
        }

        if ('admin_offers_new_variant' === $request->get('_route')) {
            $copy = $this->offers->createVariant($offer);

            if ($edition instanceof Edition) {
                $copy->setEdition($edition);
            }

            $this->denyAccessUnlessGranted('view', $offer);
            $this->denyAccessUnlessGranted('create', $copy);

            $this->entityManager->persist($copy);

            return $copy;
        }

        return $offer;
    }
}
