<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Doctrine\Persistence\ManagerRegistry;
use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\CoreBundle\ApplicationSystem\ApplicationSystems;
use Ferienpass\CoreBundle\Entity\HostConsent;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Export\ParticipantList\PdfExport;
use Ferienpass\CoreBundle\Export\ParticipantList\WordExport;
use Ferienpass\CoreBundle\Export\ParticipantList\XlsxExport;
use Ferienpass\CoreBundle\Repository\HostConsentRepository;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Requirement\Requirement;
use Symfony\Component\Security\Http\Attribute\CurrentUser;

#[Route('/angebote/{uuid}/zuordnen', requirements: ['uuid' => Requirement::UUID])]
class OfferAssignController extends AbstractController
{
    public function __construct(private readonly PdfExport $pdfExport, private readonly WordExport $wordExport, private readonly XlsxExport $xlsxExport)
    {
    }

    #[Route('', name: 'admin_offer_assign')]
    public function __invoke(string $uuid, #[CurrentUser] User $user, Request $request, OfferRepositoryInterface $offers, Session $session, ManagerRegistry $doctrine, HostConsentRepository $consents, Breadcrumb $breadcrumb, ApplicationSystems $applicationSystems, #[Autowire(param: 'ferienpass_admin.privacy_consent_text')] string $consentText): Response
    {
        /** @var OfferInterface $offer */
        if (!($offer = $offers->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        if ($offer->getEdition()->hostsCanAssign()) {
            $this->denyAccessUnlessGranted('participants.view', $offer);
        } else {
            $this->denyAccessUnlessGranted('ROLE_ADMIN');
        }

        if ('' !== $consentText && !$consents->findValid($user) instanceof HostConsent) {
            return $this->render('@FerienpassAdmin/page/missing_privacy_statement.html.twig', [
                'breadcrumb' => $breadcrumb->generate(
                    ['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]],
                    [$offer->getEdition()->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]],
                    $offer->getName(),
                    'Anmeldungen'
                ),
            ]);
        }

        $toggleMode = null;
        $autoAssign = $session->get('admin--auto-assign', false);
        if ('firstcome' === $applicationSystems->findApplicationSystem($offer)?->getType()) {
            $toggleMode = $this->createFormBuilder(['auto' => $autoAssign])
                ->add('auto', CheckboxType::class, ['false_values' => ['']])
                ->getForm()
            ;

            $toggleMode->handleRequest($request);
            if ($toggleMode->isSubmitted() && $toggleMode->isValid()) {
                $autoAssign = $toggleMode->get('auto')->getData();

                $session->set('admin--auto-assign', $autoAssign);
            }
        }

        return $this->render('@FerienpassAdmin/page/offers/assign.html.twig', [
            'offer' => $offer,
            'toggleMode' => $toggleMode,
            'autoAssign' => $autoAssign,
            'hasWordExport' => $this->wordExport->hasTemplate(),
            'breadcrumb' => $breadcrumb->generate(
                ['offers.title', ['route' => 'admin_offers_index', 'routeParameters' => array_filter(['edition' => $offer->getEdition()->getAlias()])]],
                [$offer->getEdition()->getName(), ['route' => 'admin_offers_index', 'routeParameters' => ['edition' => $offer->getEdition()->getAlias()]]],
                $offer->getName(),
                'Anmeldungen'
            ),
        ]);
    }

    #[Route('.pdf', name: 'admin_offer_assign_pdf')]
    public function pdf(string $uuid, OfferRepositoryInterface $offerRepository): Response
    {
        if (!($offer = $offerRepository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('participants.view', $offer);

        $path = $this->pdfExport->generate($offer);

        return $this->file($path, 'teilnahmeliste.pdf');
    }

    #[Route('.docx', name: 'admin_offer_assign_docx')]
    public function docx(string $uuid, OfferRepositoryInterface $offerRepository): Response
    {
        if (!($offer = $offerRepository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('participants.view', $offer);

        $path = $this->wordExport->generate($offer);

        return $this->file($path, 'teilnahmeliste.docx');
    }

    #[Route('.xlsx', name: 'admin_offer_assign_xlsx')]
    public function xlsx(string $uuid, OfferRepositoryInterface $offerRepository): Response
    {
        if (!($offer = $offerRepository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('participants.view', $offer);

        $path = $this->xlsxExport->generate($offer);

        return $this->file($path, 'teilnahmeliste.xlsx');
    }

    #[Route('/export', name: 'admin_offer_assign_export_modal')]
    public function exportModal(string $uuid, OfferRepositoryInterface $offerRepository): Response
    {
        if (!($offer = $offerRepository->findByUuid($uuid)) instanceof OfferInterface) {
            throw $this->createNotFoundException();
        }

        $this->denyAccessUnlessGranted('participants.view', $offer);

        return $this->render('@FerienpassAdmin/page/offers/assign_export_modal.html.twig', [
            'offer' => $offer,
        ]);
    }
}
