<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\AdminBundle\Form\Filter\ConsentsFilter;
use Ferienpass\CoreBundle\Entity\Consent;
use Ferienpass\CoreBundle\Export\Consent\ConsentSheetExport;
use Ferienpass\CoreBundle\Repository\ConsentRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\UX\Turbo\TurboBundle;

#[IsGranted('ROLE_ADMIN')]
#[Route('/einwilligungen')]
final class ConsentsController extends AbstractController
{
    #[Route('', name: 'admin_consents_index')]
    public function index(ConsentRepository $repository, Breadcrumb $breadcrumb): Response
    {
        $qb = $repository->createQueryBuilder('i');
        $qb->leftJoin('i.user', 'user');
        $qb->orderBy('i.createdAt', 'DESC');

        return $this->render('@FerienpassAdmin/page/consents/index.html.twig', [
            'qb' => $qb,
            'filterType' => ConsentsFilter::class,
            'searchable' => ['user.firstname', 'user.lastname'],
            'breadcrumb' => $breadcrumb->generate(['tools.title', ['route' => 'admin_tools']], 'signatures.title'),
        ]);
    }

    #[Route('/{id}/löschen', name: 'admin_consents_delete', requirements: ['id' => '\d+'])]
    public function delete(Consent $item, Request $request, EntityManagerInterface $entityManager): Response
    {
        $this->denyAccessUnlessGranted('delete', $item);

        $form = $this->createForm(FormType::class, options: [
            'action' => $this->generateUrl('admin_consents_delete', ['id' => $item->getId()]),
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $deletedId = $item->getId();

            $entityManager->remove($item);
            $entityManager->flush();

            if (TurboBundle::STREAM_FORMAT === $request->getPreferredFormat()) {
                // If the request comes from Turbo, set the content type as text/vnd.turbo-stream.html and only send the HTML to update
                $request->setRequestFormat(TurboBundle::STREAM_FORMAT);

                return $this->renderBlock('@FerienpassAdmin/page/consents/index.html.twig', 'deleted_stream', ['id' => $deletedId]);
            }

            return $this->redirectToRoute('admin_consents_index');
        }

        return $this->render('@FerienpassAdmin/page/consents/delete.html.twig', [
            'item' => $item,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/widerrufen', name: 'admin_consents_revoke', requirements: ['id' => '\d+'])]
    public function revoke(Consent $item, Request $request, EntityManagerInterface $entityManager): Response
    {
        $this->denyAccessUnlessGranted('revoke', $item);

        $form = $this->createForm(FormType::class, options: [
            'action' => $this->generateUrl('admin_consents_revoke', ['id' => $item->getId()]),
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $item->revoke();

            $entityManager->flush();

            //            if (TurboBundle::STREAM_FORMAT === $request->getPreferredFormat()) {
            //                // If the request comes from Turbo, set the content type as text/vnd.turbo-stream.html and only send the HTML to update
            //                $request->setRequestFormat(TurboBundle::STREAM_FORMAT);
            //
            //                return $this->renderBlock('@FerienpassAdmin/page/signatures/index.html.twig', 'updated_stream', ['item' => $item]);
            //            }

            return $this->redirectToRoute('admin_consents_index');
        }

        return $this->render('@FerienpassAdmin/page/consents/revoke.html.twig', [
            'item' => $item,
            'form' => $form,
        ]);
    }

    #[Route('/{id}/dokument', name: 'admin_consents_sheet', requirements: ['id' => '\d+'])]
    public function sheet(Consent $item, ConsentSheetExport $export): Response
    {
        $this->denyAccessUnlessGranted('view', $item);

        $path = $export->generate($item);

        return $this->file($path, \sprintf('einwilligung-%d.pdf', $item->getId()));
    }
}
