<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Controller\Page;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\AdminBundle\Breadcrumb\Breadcrumb;
use Ferienpass\AdminBundle\Export\ExportQueryBuilderInterface;
use Ferienpass\AdminBundle\Form\EditAccountType;
use Ferienpass\AdminBundle\Form\Filter\AccountsFilter;
use Ferienpass\AdminBundle\Form\FormScreenFactory;
use Ferienpass\AdminBundle\Form\HandleFormTrait;
use Ferienpass\CoreBundle\Entity\User;
use Ferienpass\CoreBundle\Entity\UserActivity;
use Ferienpass\CoreBundle\Message\UnblockEmail;
use Ferienpass\CoreBundle\Repository\SentMessageRepository;
use Ferienpass\CoreBundle\Repository\UserRepository;
use Knp\Menu\FactoryInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\TaggedLocator;
use Symfony\Component\DependencyInjection\ServiceLocator;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Attribute\MapQueryParameter;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Requirement\Requirement;
use Symfony\Component\Security\Http\Attribute\IsGranted;
use Symfony\UX\Turbo\TurboBundle;

#[IsGranted('ROLE_ADMIN')]
#[Route('/accounts')]
final class AccountsController extends AbstractController
{
    use HandleFormTrait;

    public const array ROLES = [
        'eltern' => 'ROLE_MEMBER',
        'veranstaltende' => 'ROLE_HOST',
        'admins' => 'ROLE_ADMIN',
    ];

    public function __construct(#[TaggedLocator(ExportQueryBuilderInterface::class, defaultIndexMethod: 'getFormat')] private readonly ServiceLocator $exporters)
    {
    }

    #[Route('/{!role?eltern}', name: 'admin_accounts_index', requirements: ['role' => '\w+'])]
    public function index(string $role, UserRepository $repository, Breadcrumb $breadcrumb, FactoryInterface $menuFactory): Response
    {
        if (!\in_array($role, array_keys(self::ROLES), true)) {
            throw $this->createNotFoundException('The role does not exist');
        }

        // Only super admins can edit admins
        if (!$this->isGranted('ROLE_SUPER_ADMIN') && 'ROLE_ADMIN' === self::ROLES[$role]) {
            throw $this->createAccessDeniedException();
        }

        $actualRole = self::ROLES[$role];

        $qb = $repository->createQueryBuilder('i')
            ->where("JSON_SEARCH(i.roles, 'one', :role) IS NOT NULL")
            ->setParameter('role', $actualRole)
        ;

        if ('ROLE_HOST' === $actualRole) {
            $qb
                ->leftJoin('i.hostAssociations', 'ha')
                ->leftJoin('ha.host', 'h')
            ;
        }

        // Solve N+1
        $qb
            ->leftJoin('i.blockedContact', 'bc')
            ->addSelect('bc')
            ->leftJoin('i.debtor', 'd')
            ->addSelect('d')
            ->leftJoin('i.activity', 'l')
            ->addSelect('l')
            ->leftJoin('i.participants', 'p')
            ->addSelect('p')
            ->leftJoin('p.attendances', 'a')
            ->addSelect('a')
        ;

        $nav = $menuFactory->createItem('accounts.roles');
        foreach (self::ROLES as $slug => $r) {
            if ('ROLE_ADMIN' === $r && !$this->isGranted('ROLE_SUPER_ADMIN')) {
                continue;
            }

            $nav->addChild('accounts.'.$r, [
                'route' => 'admin_accounts_index',
                'routeParameters' => ['role' => $slug],
                'current' => $slug === $role,
            ]);
        }

        $searchable = ['firstname', 'lastname', 'email'];
        if ('ROLE_HOST' === $actualRole) {
            $searchable[] = 'h.name';
        }

        return $this->render('@FerienpassAdmin/page/accounts/index.html.twig', [
            'qb' => $qb,
            'filterType' => AccountsFilter::class,
            'exports' => array_keys($this->exporters->getProvidedServices()),
            'role' => $actualRole,
            'searchable' => $searchable,
            'createUrl' => $this->generateUrl('admin_accounts_create', ['role' => $role]),
            'headline' => 'accounts.'.$actualRole,
            'aside_nav' => $nav,
            'breadcrumb' => $breadcrumb->generate('accounts.title', 'accounts.'.self::ROLES[$role]),
        ]);
    }

    #[Route('/{role}/export.{format}', name: 'admin_accounts_export', requirements: ['role' => '\w+', 'format' => '\w+'])]
    public function export(string $role, UserRepository $repository, string $format)
    {
        if (!$this->exporters->has($format)) {
            throw $this->createNotFoundException();
        }

        if (!\in_array($role, array_keys(self::ROLES), true)) {
            throw $this->createNotFoundException('The role does not exist');
        }

        // Only super admins can edit admins
        if (!$this->isGranted('ROLE_SUPER_ADMIN') && 'ROLE_ADMIN' === self::ROLES[$role]) {
            throw $this->createAccessDeniedException();
        }

        $actualRole = self::ROLES[$role];

        $qb = $repository->createQueryBuilder('i')
            ->where("JSON_SEARCH(i.roles, 'one', :role) IS NOT NULL")
            ->setParameter('role', $actualRole)
        ;

        $exporter = $this->exporters->get($format);

        return $this->file($exporter->generate($qb), "accounts-$role.$format");
    }

    #[Route('/{uuid}/info', name: 'admin_accounts_show', requirements: ['uuid' => Requirement::UUID])]
    public function show(?User $user, #[MapQueryParameter(name: 'kommentare')] ?bool $commentsOnly, Breadcrumb $breadcrumb, SentMessageRepository $messages): Response
    {
        $this->denyAccessUnlessGranted('view', $user);

        return $this->render('@FerienpassAdmin/page/accounts/show.html.twig', [
            'item' => $user,
            'messages' => $messages->findByUser($user),
            'commentsOnly' => (bool) $commentsOnly,
            'breadcrumb' => $breadcrumb->generate(['accounts.title', ['route' => 'admin_accounts_index'], $user->getName()]),
        ]);
    }

    #[Route('/{role}/neu/{section?default}', name: 'admin_accounts_create', requirements: ['role' => '\w+'])]
    #[Route('/{uuid}/{section?default}', name: 'admin_accounts_edit', requirements: ['uuid' => Requirement::UUID], priority: -1)]
    public function edit(?User $user, ?string $role, string $section, Request $request, EntityManagerInterface $entityManager, EventDispatcherInterface $dispatcher, Breadcrumb $breadcrumb, FormScreenFactory $forms): Response
    {
        if ($user instanceof User) {
            $this->denyAccessUnlessGranted('edit', $user);
        }

        if (!$user instanceof User && $role && isset(self::ROLES[$role])) {
            $roleName = self::ROLES[$role];
            $user = new User();
            $user->setRoles([$roleName]);
        }

        $role ??= array_find_key(self::ROLES, fn ($r) => $r === $user->getAccountRoles()[0] ?? null) ?? null;
        $roleName ??= self::ROLES[$role] ?? null;

        $form = $this->createForm(EditAccountType::class, $user, $formOptions = [
            'display_section' => $section,
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $this->handleSubmit($form, $dispatcher, $request);

            $isNew = !$user->getId();

            $entityManager->flush();

            if ($isNew) {
                return $this->redirectToRoute('admin_accounts_password', ['uuid' => $user->getUuid()]);
            }

            return $this->redirectToRoute('admin_accounts_edit', ['uuid' => $user->getUuid(), 'section' => $section]);
        }

        $breadcrumbTitle = $user->getId() ? \sprintf('%s (bearbeiten)', $user->getName()) : 'accounts.new';

        $breadcrumb = $breadcrumb->generate(
            ['accounts.title', ['route' => 'admin_accounts_index', 'routeParameters' => array_filter(['role' => $role])]],
            $role ? ['accounts.'.$roleName, ['route' => 'admin_accounts_index', 'routeParameters' => ['role' => $role]]] : null,
            $breadcrumbTitle
        );

        return $this->render('@FerienpassAdmin/page/accounts/edit.html.twig', [
            'item' => $user,
            'form' => $form,
            'formOptions' => $formOptions,
            'formScreen' => $forms->screen(EditAccountType::class, $user),
            'section' => $section,
            'headline' => $user->getId() ? $user->getName() : 'accounts.new',
            'breadcrumb' => $breadcrumb,
        ]);
    }

    #[Route('/{uuid}/löschen', name: 'admin_accounts_delete', requirements: ['uuid' => Requirement::UUID])]
    public function delete(User $item, Request $request, EntityManagerInterface $entityManager): Response
    {
        $this->denyAccessUnlessGranted('delete', $item);

        $form = $this->createForm(FormType::class, options: [
            'action' => $this->generateUrl('admin_accounts_delete', ['uuid' => $item->getUuid()]),
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $createdByActivities = $entityManager->getRepository(UserActivity::class)->findBy(['createdBy' => $item]);
            foreach ($createdByActivities as $activity) {
                $entityManager->remove($activity);
            }

            foreach ($item->getParticipants() as $participant) {
                /** @var $attendance */
                foreach ($participant->getAttendances() as $attendance) {
                    foreach ($attendance->getPaymentItems() as $paymentItem) {
                        $paymentItem->removeAttendanceAssociation();
                    }
                }
            }

            $deletedId = $item->getId();

            $entityManager->remove($item);
            $entityManager->flush();

            if (TurboBundle::STREAM_FORMAT === $request->getPreferredFormat()) {
                // If the request comes from Turbo, set the content type as text/vnd.turbo-stream.html and only send the HTML to update
                $request->setRequestFormat(TurboBundle::STREAM_FORMAT);

                return $this->renderBlock('@FerienpassAdmin/page/accounts/index.html.twig', 'deleted_stream', ['id' => $deletedId]);
            }

            return $this->redirectToRoute('admin_accounts_index');
        }

        return $this->render('@FerienpassAdmin/page/accounts/delete.html.twig', [
            'item' => $item,
            'form' => $form,
        ]);
    }

    #[Route('/{uuid}/passwort', name: 'admin_accounts_password', requirements: ['uuid' => Requirement::UUID])]
    public function password(User $item, Request $request, EntityManagerInterface $entityManager, UserPasswordHasherInterface $passwordHasher): Response
    {
        $this->denyAccessUnlessGranted('password', $item);

        $session = $request->getSession();
        $form = $this->createForm(FormType::class, options: [
            'action' => $this->generateUrl('admin_accounts_password', ['uuid' => $item->getUuid()]),
        ]);

        if ($session->has('account_password_tmp')) {
            $password = $session->get('account_password_tmp');
            $session->remove('account_password_tmp');

            return $this->render('@FerienpassAdmin/page/accounts/password.html.twig', [
                'item' => $item,
                'password' => $password,
            ]);
        }

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $password = $this->generatePassword();
            $encodedPassword = $passwordHasher->hashPassword($item, $password);

            $session->set('account_password_tmp', $password);
            $item->setPassword($encodedPassword);
            $item->setModifiedAt();

            $entityManager->flush();

            return $this->redirectToRoute('admin_accounts_password', ['uuid' => $item->getUuid()]);
        }

        return $this->render('@FerienpassAdmin/page/accounts/password.html.twig', [
            'item' => $item,
            'form' => $form,
        ]);
    }

    #[Route('/{uuid}/email-freischalten', name: 'admin_accounts_unblock', requirements: ['uuid' => Requirement::UUID])]
    public function unblock(User $item, Request $request, MessageBusInterface $commandBus): Response
    {
        $this->denyAccessUnlessGranted('unblock', $item);

        if (!$item->isBlockedEmail()) {
            return $this->render('@FerienpassAdmin/page/accounts/unblock.html.twig', [
                'item' => $item,
            ]);
        }

        $form = $this->createForm(FormType::class, options: [
            'action' => $this->generateUrl('admin_accounts_unblock', ['uuid' => $item->getUuid()]),
        ]);

        $form->handleRequest($request);
        if ($form->isSubmitted() && $form->isValid()) {
            $commandBus->dispatch(new UnblockEmail($item->getEmail(), $item->getId()));

            return $this->redirectToRoute('admin_accounts_unblock', ['uuid' => $item->getUuid()]);
        }

        return $this->render('@FerienpassAdmin/page/accounts/unblock.html.twig', [
            'item' => $item,
            'form' => $form,
        ]);
    }

    private function generatePassword(): string
    {
        $password = '';
        $keyspace = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';

        $max = mb_strlen($keyspace, '8bit') - 1;
        for ($i = 0; $i < 12; ++$i) {
            $password .= $keyspace[random_int(0, $max)];
        }

        return $password;
    }
}
