<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Components;

use Contao\StringUtil;
use Doctrine\ORM\NoResultException;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;
use Ferienpass\CoreBundle\Notification\MailingNotification;
use Ferienpass\CoreBundle\Notifier\Notifier;
use Ferienpass\CoreBundle\Repository\EditionRepository;
use Ferienpass\CoreBundle\Repository\HostRepository;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Ferienpass\CoreBundle\Repository\ParticipantRepositoryInterface;
use Ferienpass\CoreBundle\Repository\UserRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveListener;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;
use Symfony\UX\LiveComponent\Metadata\UrlMapping;
use Symfony\UX\LiveComponent\ValidatableComponentTrait;
use Symfony\UX\TwigComponent\Attribute\ExposeInTemplate;
use Symfony\UX\TwigComponent\Attribute\PostMount;
use Twig\Environment;

#[AsLiveComponent(route: 'live_component_admin')]
class OffersAssign extends AbstractController
{
    use ComponentToolsTrait;
    use DefaultActionTrait;
    use ValidatableComponentTrait;

    #[LiveProp]
    public ?Edition $edition = null;

    #[LiveProp(onUpdated: 'onOfferUpdated', url: new UrlMapping(as: 'angebot'))]
    public ?OfferInterface $offer = null;

    #[LiveProp]
    public ?ParticipantInterface $participant = null;

    #[LiveProp(writable: true)]
    public string $query = '';

    public function __construct(private readonly EditionRepository $editions, private readonly OfferRepositoryInterface $offers, private readonly ParticipantRepositoryInterface $participants, private readonly UserRepository $users, private readonly HostRepository $hosts, private readonly Environment $twig, private readonly RequestStack $requestStack, private readonly NormalizerInterface $normalizer, private readonly Notifier $notifier, private readonly MailingNotification $mailingNotification, #[Autowire(param: 'ferienpass.preferred_postal_codes')] private readonly array $preferredPostalCodes)
    {
    }

    #[ExposeInTemplate]
    public function offers(): iterable
    {
        $cancelled = OfferInterface::STATE_CANCELLED;

        /** @var QueryBuilder $qb */
        $qb = $this->offers->createQueryBuilder('o');
        $qb
            ->addSelect('sum(CASE WHEN a.status = :status_waiting THEN 1 ELSE 0 END) AS HIDDEN countWaiting')
            ->addSelect('CASE WHEN o.maxParticipants > 0 AND sum(CASE WHEN a.status = :status_confirmed THEN 1 ELSE 0 END) < o.maxParticipants AND sum(CASE WHEN a.status = :status_waitlisted THEN 1 ELSE 0 END) > 0 THEN 1 ELSE 0 END AS HIDDEN falseWaitlisted')
            ->leftJoin('o.hosts', 'h')
            ->leftJoin('o.dates', 'd')
            ->leftJoin('o.attendances', 'a')
            ->addSelect('a')
            ->leftJoin('o.activity', 'l')
            ->addSelect('l')
            ->leftJoin('o.variants', 'variants')
            ->addSelect('variants')
            ->leftJoin('a.participant', 'participant')
            ->addSelect('participant')
            ->leftJoin('participant.attendances', 'participant_attendance')
            ->addSelect('participant_attendance')
            ->leftJoin('participant_attendance.offer', 'participant_offer')
            ->addSelect('participant_offer')
            ->andWhere('o.requiresApplication = 1')
            ->andWhere('o.onlineApplication = 1')
            ->addOrderBy("CASE WHEN JSON_CONTAINS_PATH(o.status, 'one', '$.$cancelled') = 1 THEN 0 ELSE 1 END", 'DESC')
            ->addOrderBy('countWaiting', 'DESC')
            ->addOrderBy('falseWaitlisted', 'DESC')
            ->addOrderBy('d.begin')
            ->addGroupBy('o.id')
            ->addGroupBy('d.begin')
            ->setParameter('status_waiting', Attendance::STATUS_WAITING)
            ->setParameter('status_confirmed', Attendance::STATUS_CONFIRMED)
            ->setParameter('status_waitlisted', Attendance::STATUS_WAITLISTED)
        ;

        $this->addQueryBuilderSearch($qb);

        if ($this->edition instanceof Edition) {
            $qb->andWhere('o.edition = :edition')->setParameter('edition', $this->edition);
        }

        if ($this->participant instanceof ParticipantInterface) {
            $qb
                ->innerJoin('a.participant', 'p', Join::WITH, 'p.id = :participant')
                ->setParameter('participant', $this->participant->getId())
            ;
        }

        return $qb->getQuery()->getResult();
    }

    #[ExposeInTemplate]
    public function participants(): iterable
    {
        /** @var QueryBuilder $qb */
        $qb = $this->participants->createQueryBuilder('p');
        $qb
            ->innerJoin('p.attendances', 'a')
            ->addGroupBy('p')
        ;

        if ($this->offer instanceof OfferInterface) {
            $qb
                ->innerJoin('a.offer', 'o', Join::WITH, 'o.id = :offer')
                ->addGroupBy('a')
                ->setParameter('offer', $this->offer)
            ;
        } else {
            $qb
                ->innerJoin('a.offer', 'o')
            ;
        }

        if ($this->offer instanceof OfferInterface) {
            $qb
                ->addOrderBy('a.userPriority', 'ASC')
                ->addOrderBy('a.status')
                ->addOrderBy('a.createdAt', 'ASC')
            ;
        }

        if (!$this->offer instanceof OfferInterface) {
            $cancelled = OfferInterface::STATE_CANCELLED;

            $qb
                ->addSelect("sum(CASE WHEN JSON_CONTAINS_PATH(o.status, 'one', '$.$cancelled') = 1 AND a.status = :status_waiting THEN 1 ELSE 0 END) AS HIDDEN countWaiting")
                ->addSelect('sum(CASE WHEN a.status = :status_confirmed THEN 1 ELSE 0 END) AS HIDDEN countConfirmed')
                ->addOrderBy('countWaiting', 'DESC')
                ->addOrderBy('countConfirmed', 'DESC')
                ->setParameter('status_waiting', Attendance::STATUS_WAITING)
                ->setParameter('status_confirmed', Attendance::STATUS_CONFIRMED)
            ;
        }

        if ($this->edition instanceof Edition) {
            $qb->innerJoin('o.edition', 'e', Join::WITH, 'e = :edition')->setParameter('edition', $this->edition);
        }

        //        $qb->addSelect('sum(CASE WHEN a.status = :status_waiting THEN 1 ELSE 0 END) AS HIDDEN countAttendances')
        //            ->leftJoin('o.hosts', 'h')
        //            ->leftJoin('o.dates', 'd')
        //            ->leftJoin('o.attendances', 'a')
        //            ->andWhere('o.requiresApplication = 1')
        //            ->andWhere('o.onlineApplication = 1')
        //            ->addOrderBy('countAttendances', 'DESC')
        //            ->addOrderBy('d.begin')
        //            ->addGroupBy('o.id')
        //            ->addGroupBy('d.begin')
        //            ->setParameter('status_waiting', Attendance::STATUS_WAITING)
        //        ;

        return $qb->getQuery()->getResult();
    }

    #[ExposeInTemplate]
    public function maxApplications(): int
    {
        /** @var QueryBuilder $qb */
        $qb = $this->participants->createQueryBuilder('p');

        $qb
            ->innerJoin('p.attendances', 'a')
            ->innerJoin('a.offer', 'o')
            ->select('sum(CASE WHEN a.status = :status_withdrawn OR a.status = :status_unfulfilled THEN 0 ELSE 1 END) AS countAttendances')
            ->addGroupBy('p')
            ->addOrderBy('countAttendances', 'DESC')
            ->andWhere('a.status <> :status_withdrawn')
            ->andWhere('a.status <> :status_unfulfilled')
            ->setParameter('status_withdrawn', Attendance::STATUS_WITHDRAWN)
            ->setParameter('status_unfulfilled', Attendance::STATUS_UNFULFILLED)
        ;

        if ($this->edition instanceof Edition) {
            $qb->innerJoin('o.edition', 'e', Join::WITH, 'e.id = :edition')->setParameter('edition', $this->edition->getId());
        }

        try {
            $count = $qb->getQuery()->setMaxResults(1)->getSingleScalarResult();
        } catch (NoResultException) {
            return 0;
        }

        return max((int) $count, 1);
    }

    #[LiveListener('open')]
    public function openOffer(#[LiveArg('offer')] int $offerId): void
    {
        $this->offer = $this->offers->find($offerId);
    }

    #[LiveListener('selectParticipant')]
    public function selectParticipant(#[LiveArg('participant')] int $participantId): void
    {
        if (!$this->participant instanceof ParticipantInterface || $participantId !== $this->participant->getId()) {
            $this->participant = $this->participants->find($participantId);
        } else {
            $this->participant = null;
        }
    }

    #[ExposeInTemplate]
    public function showPostalCode(): bool
    {
        return (bool) $this->preferredPostalCodes;
    }

    public function onOfferUpdated($previous): void
    {
    }

    #[PostMount]
    public function postMount(): void
    {
        // Fix invalid entity when ?angebot=
        if (null === $this->offer?->getId()) {
            $this->offer = null;
        }
    }

    private function addQueryBuilderSearch(QueryBuilder $qb): void
    {
        $where = $qb->expr()->orX();
        foreach (array_filter(StringUtil::trimsplit(' ', $this->query)) as $j => $token) {
            foreach (['o.name', 'h.name'] as $field) {
                $where->add("$field LIKE :query_$j");
            }

            $qb->setParameter("query_$j", "%{$token}%");
        }

        if ($where->count()) {
            $qb->andWhere($where);
        }
    }
}
