<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Components;

use Doctrine\ORM\EntityManagerInterface;
use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Offer\Base;
use Ferienpass\CoreBundle\Entity\Offer\OfferInterface;
use Ferienpass\CoreBundle\Entity\Participant\ParticipantInterface;
use Ferienpass\CoreBundle\Repository\ParticipantRepositoryInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\Workflow\Exception\NotEnabledTransitionException;
use Symfony\Component\Workflow\WorkflowInterface;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveArg;
use Symfony\UX\LiveComponent\Attribute\LiveListener;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\DefaultActionTrait;

#[AsLiveComponent(route: 'live_component_admin')]
class OfferAssign extends AbstractController
{
    use DefaultActionTrait;

    /** @var Base */
    #[LiveProp(updateFromParent: true)]
    public OfferInterface $offer;

    #[LiveProp(updateFromParent: true)]
    public ?ParticipantInterface $participant = null;

    #[LiveProp(writable: true)]
    public bool $autoAssign = false;

    public function __construct(private readonly ParticipantRepositoryInterface $participants)
    {
    }

    #[LiveListener('selectParticipant')]
    public function selectParticipant(#[LiveArg] int $participant): void
    {
        $this->participant = $this->participants->find($participant);
    }

    #[LiveAction]
    public function confirmAllWaiting(EntityManagerInterface $em, WorkflowInterface $attendanceStateMachine): void
    {
        $attendances = $this->offer->getAttendancesWaiting();

        $lastAttendance = $this->offer->getAttendancesConfirmed()->last();
        $sorting = $lastAttendance ? $lastAttendance->getSorting() : 0;

        foreach ($attendances as $attendance) {
            try {
                $attendanceStateMachine->apply($attendance, Attendance::TRANSITION_CONFIRM, [
                    'notify' => $this->autoAssign,
                    'reorder' => $this->autoAssign,
                    'commit' => $this->autoAssign,
                ]);

                $attendance->setSorting($sorting += 128);
            } catch (NotEnabledTransitionException) {
            }
        }

        $em->flush();

        // Refresh because we changed the order of the attendances off-reference and that messes re-render
        $em->refresh($this->offer);
    }

    #[LiveAction]
    public function waitlistAllWaiting(EntityManagerInterface $em, WorkflowInterface $attendanceStateMachine): void
    {
        $attendances = $this->offer->getAttendancesWaiting();

        $lastAttendance = $this->offer->getAttendancesWaitlisted()->last();
        $sorting = $lastAttendance ? $lastAttendance->getSorting() : 0;

        foreach ($attendances as $attendance) {
            try {
                $attendanceStateMachine->apply($attendance, Attendance::TRANSITION_WAITLIST, [
                    'notify' => $this->autoAssign,
                    'reorder' => $this->autoAssign,
                    'commit' => $this->autoAssign,
                ]);

                $attendance->setSorting($sorting += 128);
            } catch (NotEnabledTransitionException) {
            }
        }

        $em->flush();

        // Refresh because we changed the order of the attendances off-reference and that messes re-render
        $em->refresh($this->offer);
    }

    #[LiveListener('statusChanged')]
    public function changeStatus(#[LiveArg] Attendance $attendance, #[LiveArg] string $newStatus, #[LiveArg] int $newIndex, EntityManagerInterface $em, WorkflowInterface $attendanceStateMachine): void
    {
        $this->denyAccessUnlessGranted('participants.view', $attendance->getOffer());

        if (!$attendance->getParticipant() instanceof ParticipantInterface) {
            return;
        }

        if ($newStatus === $attendance->getStatus()) {
            return;
        }

        $transition = match ($newStatus) {
            Attendance::STATUS_CONFIRMED => Attendance::TRANSITION_CONFIRM,
            Attendance::STATUS_WAITLISTED => Attendance::TRANSITION_WAITLIST,
            Attendance::STATUS_WITHDRAWN => Attendance::TRANSITION_WITHDRAW,
            Attendance::STATUS_WAITING => Attendance::TRANSITION_RESET,
            Attendance::STATUS_REJECTED => Attendance::TRANSITION_REJECT,
            default => null,
        };

        if (null === $transition) {
            return;
        }

        try {
            $attendanceStateMachine->apply($attendance, $transition, [
                'notify' => $this->autoAssign,
                'reorder' => $this->autoAssign,
                'commit' => $this->autoAssign,
            ]);

            $this->reorderList($attendance, $newIndex);

            $em->flush();

            // Refresh because we changed the order of the attendances off-reference and that messes re-render
            $em->refresh($this->offer);
        } catch (NotEnabledTransitionException) {
        }
    }

    #[LiveAction]
    public function change(#[LiveArg] Attendance $attendance, #[LiveArg] string $transition, EntityManagerInterface $em, WorkflowInterface $attendanceStateMachine): void
    {
        $this->denyAccessUnlessGranted('participants.view', $attendance->getOffer());

        if (!$attendance->getParticipant() instanceof ParticipantInterface) {
            return;
        }

        try {
            $attendanceStateMachine->apply($attendance, $transition, [
                'notify' => $this->autoAssign,
                'reorder' => $this->autoAssign,
                'commit' => $this->autoAssign,
            ]);

            $this->reorderList($attendance);

            $em->flush();

            // Refresh because we changed the order of the attendances off-reference and that messes re-render
            $em->refresh($this->offer);
        } catch (NotEnabledTransitionException) {
        }
    }

    #[LiveListener('indexUpdated')]
    public function changeIndex(#[LiveArg] Attendance $attendance, #[LiveArg] int $newIndex, EntityManagerInterface $em): void
    {
        $this->denyAccessUnlessGranted('participants.view', $attendance->getOffer());

        $this->reorderList($attendance, $newIndex);

        $em->flush();

        // Refresh because we changed the order of the attendances off-reference and that messes re-render
        $em->refresh($this->offer);
    }

    private function reorderList(Attendance $attendance, ?int $newIndex = null): void
    {
        /** @var Attendance[] $attendances */
        $attendances = array_values($attendance->getOffer()->getAttendancesWithStatus($attendance->getStatus())->toArray());
        $fromIndex = array_search($attendance, $attendances, true);

        if (null === $newIndex) {
            $newIndex = \count($attendances);
        }

        array_splice($attendances, $newIndex, 0, array_splice($attendances, $fromIndex, 1));

        $i = 0;
        foreach ($attendances as $a) {
            $a->setSorting(++$i * 128);
        }
    }
}
