<?php

declare(strict_types=1);

/*
 * This file is part of the Ferienpass package.
 *
 * (c) Richard Henkenjohann <richard@ferienpass.online>
 *
 * For more information visit the project website <https://ferienpass.online>
 * or the documentation under <https://docs.ferienpass.online>.
 */

namespace Ferienpass\AdminBundle\Components;

use Ferienpass\CoreBundle\Entity\Attendance;
use Ferienpass\CoreBundle\Entity\Edition;
use Ferienpass\CoreBundle\Facade\DecisionsFacade;
use Ferienpass\CoreBundle\Message\CommitAttendances as CommitAttendancesMessage;
use Ferienpass\CoreBundle\Messenger\UuidStamp;
use Ferienpass\CoreBundle\Notification\MailingNotification;
use Ferienpass\CoreBundle\Notifier\Notifier;
use Ferienpass\CoreBundle\Repository\EditionRepository;
use Ferienpass\CoreBundle\Repository\HostRepository;
use Ferienpass\CoreBundle\Repository\OfferRepositoryInterface;
use Ferienpass\CoreBundle\Repository\ParticipantRepositoryInterface;
use Ferienpass\CoreBundle\Repository\UserRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Messenger\Envelope;
use Symfony\Component\Messenger\MessageBusInterface;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;
use Symfony\Component\Uid\Uuid;
use Symfony\UX\LiveComponent\Attribute\AsLiveComponent;
use Symfony\UX\LiveComponent\Attribute\LiveAction;
use Symfony\UX\LiveComponent\Attribute\LiveProp;
use Symfony\UX\LiveComponent\ComponentToolsTrait;
use Symfony\UX\LiveComponent\DefaultActionTrait;
use Symfony\UX\LiveComponent\Metadata\UrlMapping;
use Symfony\UX\LiveComponent\ValidatableComponentTrait;
use Symfony\UX\TwigComponent\Attribute\ExposeInTemplate;
use Twig\Environment;

#[AsLiveComponent(route: 'live_component_admin')]
class CommitAttendances extends AbstractController
{
    use ComponentToolsTrait;
    use DefaultActionTrait;
    use ValidatableComponentTrait;

    #[LiveProp(writable: true, url: new UrlMapping(as: 'saison'))]
    public ?Edition $edition = null;

    #[LiveProp(writable: true)]
    public bool $notify = true;

    #[LiveProp]
    public bool $showConfirm = false;

    public function __construct(private readonly EditionRepository $editions, private readonly ParticipantRepositoryInterface $participants, private readonly UserRepository $users, private readonly HostRepository $hosts, private readonly OfferRepositoryInterface $offers, private readonly Environment $twig, private readonly RequestStack $requestStack, private readonly NormalizerInterface $normalizer, private readonly Notifier $notifier, private readonly MailingNotification $mailingNotification, private readonly DecisionsFacade $unconfirmedApplications)
    {
    }

    #[ExposeInTemplate]
    public function editionOptions(): array
    {
        $qb = $this->editions->createQueryBuilder('e');

        return $qb->getQuery()->getResult();
    }

    #[ExposeInTemplate]
    public function attendances(): array
    {
        if (!$this->edition instanceof Edition || null === $this->edition->getId()) {
            return [];
        }

        $return = [];
        /** @var Attendance $attendance */
        foreach ($this->unconfirmedApplications->attendances($this->edition) as $attendance) {
            $return[$attendance->getEmail()][$attendance->getParticipant()->getId()][] = $attendance;
        }

        return $return;
    }

    #[LiveAction]
    public function submit(): void
    {
        $this->validate();

        $this->showConfirm = true;
    }

    #[LiveAction]
    public function commit(MessageBusInterface $commandBus): RedirectResponse
    {
        if (!$this->edition instanceof Edition) {
            throw new \RuntimeException('This should not happen');
        }

        $uuid = Uuid::v4();

        $command = new CommitAttendancesMessage($this->edition->getId(), array_map(fn (Attendance $a) => $a->getId(), $this->unconfirmedApplications->attendances($this->edition)), $this->notify);
        $commandBus->dispatch(
            new Envelope($command)->with(new UuidStamp($uuid))
        );

        $this->showConfirm = false;
        $this->resetValidation();

        if ($this->notify) {
            return $this->redirectToRoute('admin_tools_dispatch', ['uuid' => (string) $uuid]);
        }

        return $this->redirectToRoute('admin_send_decisions', ['saison' => $this->edition->getId()]);
    }

    #[LiveAction]
    public function cancel(): void
    {
        $this->showConfirm = false;
    }
}
