'use strict';

import {promises as fs} from 'fs';
import path from 'path';
import {fileURLToPath} from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const SRC_DIR = path.join(__dirname, '..', 'src');
const DIST_DIR = path.join(__dirname, '..', 'dist');

async function ensureDir(dir) {
  await fs.mkdir(dir, { recursive: true });
}

async function cleanDist() {
  await fs.rm(DIST_DIR, { recursive: true, force: true });
  await ensureDir(DIST_DIR);
}

async function copyJsFiles(src, dest) {
  const entries = await fs.readdir(src, { withFileTypes: true });
  for (const entry of entries) {
    const srcPath = path.join(src, entry.name);
    const destPath = path.join(dest, entry.name);
    if (entry.isDirectory()) {
      await ensureDir(destPath);
      await copyJsFiles(srcPath, destPath);
    } else if (entry.isFile() && entry.name.endsWith('.js')) {
      await ensureDir(dest);
      await fs.copyFile(srcPath, destPath);
    }
  }
}

async function main() {
  await cleanDist();
  await copyJsFiles(SRC_DIR, DIST_DIR);
  // keep CSS build separate via npm run build:css
}

main().catch((err) => {
  console.error(err);
  process.exit(1);
});
